import React from "react";
import { Controller } from "react-hook-form";
import Select, { OnChangeValue } from "react-select";
import { startCase } from "lodash-es";
import { formatReactSelectStyles } from "../FormElement/data";

interface OptionType {
  value: string;
  label: string;
}

interface SelectInputProps {
  optionsData: OptionType[] | string[];
  getOptionLabel?: (option: OptionType) => string;
  getOptionValue?: (option: OptionType) => string;
  fieldName: string;
  fieldLabel?: string;
  placeholder?: string | boolean;
  control: any;
  required?: boolean;
  isDisabled?: boolean;
  validateFn?: any;
  onChangeFn?: (value: OnChangeValue<OptionType, boolean>) => void;
  defaultValue?: string | null;
  moreCss?: string;
}

const SelectInput: React.FC<SelectInputProps> = ({
  optionsData,
  getOptionLabel,
  getOptionValue,
  fieldName,
  fieldLabel,
  placeholder = true,
  control,
  required,
  isDisabled,
  validateFn,
  onChangeFn,
  defaultValue,
  moreCss = "border-stroke",
}) => {
  const formatOptions = (data: (OptionType | string)[]) => {
    return data.map((item) => {
      if (typeof item === "string") {
        return { value: item, label: item };
      } else {
        return item as OptionType;
      }
    });
  };

  const options = formatOptions(optionsData);

  // Find the default option object from the options list
  const defaultOption =
    options.find((option) => option.value === defaultValue) || null;

  return (
    <Controller
      name={fieldName}
      control={control}
      defaultValue={defaultOption}
      rules={{
        required: required ? "This field is required." : false,
        validate: validateFn,
      }}
      render={({ field }) => (
        <Select
          {...field}
          inputId={fieldName}
          options={options}
          getOptionLabel={getOptionLabel}
          getOptionValue={getOptionValue}
          value={field.value || defaultOption}
          placeholder={
            placeholder
              ? typeof placeholder === "string"
                ? placeholder
                : `Select ${
                    fieldLabel?.toLowerCase() || startCase(fieldName)
                  }...`
              : undefined
          }
          onChange={(value) => {
            field.onChange(value);
            onChangeFn ? onChangeFn(value) : undefined;
          }}
          isSearchable
          isClearable
          closeMenuOnSelect
          blurInputOnSelect
          className={`${moreCss} rounded border-2 bg-transparent py-2.5 px-3 text-black focus:border-primary active:border-primary disabled:cursor-default disabled:bg-whiter dark:border-form-strokedark dark:bg-form-input dark:text-white dark:focus:border-primary`}
          classNamePrefix={""}
          theme={(theme) => ({
            ...theme,
            colors: {
              ...theme.colors,
              primary25: "#eeffee",
              primary: "#fff",
            },
          })}
          styles={formatReactSelectStyles}
          isDisabled={isDisabled}
        />
      )}
    />
  );
};

export default SelectInput;
