import { jsx, jsxs, Fragment } from "react/jsx-runtime";
import { PassThrough } from "node:stream";
import { createReadableStreamFromReadable } from "@remix-run/node";
import { RemixServer, Outlet, Meta, Links, ScrollRestoration, Scripts, useNavigate, useParams, Link } from "@remix-run/react";
import { isbot } from "isbot";
import { renderToPipeableStream } from "react-dom/server";
import { Provider, useSelector } from "react-redux";
import { createSlice, configureStore } from "@reduxjs/toolkit";
import { fetchBaseQuery, createApi } from "@reduxjs/toolkit/query/react";
import * as React from "react";
import { useState, useEffect, useMemo } from "react";
import * as ToastPrimitives from "@radix-ui/react-toast";
import { cva } from "class-variance-authority";
import { X, Check, Eye, EyeClosed, Loader2, CircleCheck, TriangleAlert, Search, Heart, User, ShoppingBasket, Twitter, Facebook, Instagram, Github, ArrowRight, Star, Clock, FileText, Truck } from "lucide-react";
import { clsx } from "clsx";
import { twMerge } from "tailwind-merge";
import { Controller, useFormContext, useForm, FormProvider } from "react-hook-form";
import { startCase } from "lodash-es";
import { ErrorMessage } from "@hookform/error-message";
import PhoneInput from "react-phone-number-input/react-hook-form";
import Select from "react-select";
import StarRatings from "react-star-ratings";
import { Slot } from "@radix-ui/react-slot";
import * as DialogPrimitive from "@radix-ui/react-dialog";
const ABORT_DELAY = 5e3;
function handleRequest(request, responseStatusCode, responseHeaders, remixContext, loadContext) {
  return isbot(request.headers.get("user-agent") || "") ? handleBotRequest(
    request,
    responseStatusCode,
    responseHeaders,
    remixContext
  ) : handleBrowserRequest(
    request,
    responseStatusCode,
    responseHeaders,
    remixContext
  );
}
function handleBotRequest(request, responseStatusCode, responseHeaders, remixContext) {
  return new Promise((resolve, reject) => {
    let shellRendered = false;
    const { pipe, abort } = renderToPipeableStream(
      /* @__PURE__ */ jsx(
        RemixServer,
        {
          context: remixContext,
          url: request.url,
          abortDelay: ABORT_DELAY
        }
      ),
      {
        onAllReady() {
          shellRendered = true;
          const body = new PassThrough();
          const stream = createReadableStreamFromReadable(body);
          responseHeaders.set("Content-Type", "text/html");
          resolve(
            new Response(stream, {
              headers: responseHeaders,
              status: responseStatusCode
            })
          );
          pipe(body);
        },
        onShellError(error) {
          reject(error);
        },
        onError(error) {
          responseStatusCode = 500;
          if (shellRendered) {
            console.error(error);
          }
        }
      }
    );
    setTimeout(abort, ABORT_DELAY);
  });
}
function handleBrowserRequest(request, responseStatusCode, responseHeaders, remixContext) {
  return new Promise((resolve, reject) => {
    let shellRendered = false;
    const { pipe, abort } = renderToPipeableStream(
      /* @__PURE__ */ jsx(
        RemixServer,
        {
          context: remixContext,
          url: request.url,
          abortDelay: ABORT_DELAY
        }
      ),
      {
        onShellReady() {
          shellRendered = true;
          const body = new PassThrough();
          const stream = createReadableStreamFromReadable(body);
          responseHeaders.set("Content-Type", "text/html");
          resolve(
            new Response(stream, {
              headers: responseHeaders,
              status: responseStatusCode
            })
          );
          pipe(body);
        },
        onShellError(error) {
          reject(error);
        },
        onError(error) {
          responseStatusCode = 500;
          if (shellRendered) {
            console.error(error);
          }
        }
      }
    );
    setTimeout(abort, ABORT_DELAY);
  });
}
const entryServer = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: handleRequest
}, Symbol.toStringTag, { value: "Module" }));
const slice = createSlice({
  name: "auth",
  initialState: null,
  reducers: {
    setCredentials: (state, action) => {
      state = action.payload;
      return state;
    },
    logOut: (state) => {
      window.localStorage.setItem("authUser", "");
      state = null;
      return state;
    }
  }
});
const { setCredentials, logOut } = slice.actions;
const authReducer = slice.reducer;
const selectCurrentUser = (state) => state.auth;
const baseURL$1 = "https://oneprint.atomus.buzz/";
const baseQuery = fetchBaseQuery({
  baseUrl: baseURL$1,
  // credentials: 'include',
  prepareHeaders: (headers) => {
    const user = window.localStorage.getItem("authUser");
    const token = user && JSON.parse(user || "");
    if (token) {
      headers.set("authorization", `Bearer ${token == null ? void 0 : token.jwtToken}`);
    }
    return headers;
  }
});
const baseQueryWithReauth = async (args, api, extraOptions) => {
  let result = await baseQuery(args, api, extraOptions);
  if ("error" in result) {
    if (result.error.status === 401 || result.error.status === 403) {
      console.log("sending refresh token");
      const refreshResult = await baseQuery("/refresh", api, extraOptions);
      console.log(refreshResult);
      if (refreshResult == null ? void 0 : refreshResult.data) {
        const user = api.getState().auth;
        api.dispatch(setCredentials({ ...user, ...refreshResult.data }));
        result = await baseQuery(args, api, extraOptions);
      } else {
        api.dispatch(logOut());
      }
    }
  }
  return result;
};
const apiSlice = createApi({
  baseQuery: baseQueryWithReauth,
  tagTypes: ["Enum"],
  endpoints: () => ({})
});
const baseURL = "https://oneprint.atomus.buzz/";
const noAuth = createApi({
  reducerPath: "noAuth",
  baseQuery: fetchBaseQuery({
    baseUrl: baseURL,
    prepareHeaders: (headers, { getState }) => {
      return headers;
    }
  }),
  endpoints: () => ({})
});
const store = configureStore({
  reducer: {
    [apiSlice.reducerPath]: apiSlice.reducer,
    [noAuth.reducerPath]: noAuth.reducer,
    auth: authReducer
  },
  middleware: (getDefaultMiddleware) => getDefaultMiddleware().concat(apiSlice.middleware, noAuth.middleware)
});
const TOAST_LIMIT = 1;
const TOAST_REMOVE_DELAY = 1e6;
let count = 0;
function genId() {
  count = (count + 1) % Number.MAX_SAFE_INTEGER;
  return count.toString();
}
const toastTimeouts = /* @__PURE__ */ new Map();
const addToRemoveQueue = (toastId) => {
  if (toastTimeouts.has(toastId)) {
    return;
  }
  const timeout = setTimeout(() => {
    toastTimeouts.delete(toastId);
    dispatch({
      type: "REMOVE_TOAST",
      toastId
    });
  }, TOAST_REMOVE_DELAY);
  toastTimeouts.set(toastId, timeout);
};
const reducer = (state, action) => {
  switch (action.type) {
    case "ADD_TOAST":
      return {
        ...state,
        toasts: [action.toast, ...state.toasts].slice(0, TOAST_LIMIT)
      };
    case "UPDATE_TOAST":
      return {
        ...state,
        toasts: state.toasts.map(
          (t) => t.id === action.toast.id ? { ...t, ...action.toast } : t
        )
      };
    case "DISMISS_TOAST": {
      const { toastId } = action;
      if (toastId) {
        addToRemoveQueue(toastId);
      } else {
        state.toasts.forEach((toast2) => {
          addToRemoveQueue(toast2.id);
        });
      }
      return {
        ...state,
        toasts: state.toasts.map(
          (t) => t.id === toastId || toastId === void 0 ? {
            ...t,
            open: false
          } : t
        )
      };
    }
    case "REMOVE_TOAST":
      if (action.toastId === void 0) {
        return {
          ...state,
          toasts: []
        };
      }
      return {
        ...state,
        toasts: state.toasts.filter((t) => t.id !== action.toastId)
      };
  }
};
const listeners = [];
let memoryState = { toasts: [] };
function dispatch(action) {
  memoryState = reducer(memoryState, action);
  listeners.forEach((listener) => {
    listener(memoryState);
  });
}
function toast({ ...props }) {
  const id = genId();
  const update = (props2) => dispatch({
    type: "UPDATE_TOAST",
    toast: { ...props2, id }
  });
  const dismiss = () => dispatch({ type: "DISMISS_TOAST", toastId: id });
  dispatch({
    type: "ADD_TOAST",
    toast: {
      ...props,
      id,
      open: true,
      onOpenChange: (open) => {
        if (!open) dismiss();
      }
    }
  });
  return {
    id,
    dismiss,
    update
  };
}
function useToast() {
  const [state, setState] = React.useState(memoryState);
  React.useEffect(() => {
    listeners.push(setState);
    return () => {
      const index = listeners.indexOf(setState);
      if (index > -1) {
        listeners.splice(index, 1);
      }
    };
  }, [state]);
  return {
    ...state,
    toast,
    dismiss: (toastId) => dispatch({ type: "DISMISS_TOAST", toastId })
  };
}
function cn(...inputs) {
  return twMerge(clsx(inputs));
}
const ToastProvider = ToastPrimitives.Provider;
const ToastViewport = React.forwardRef(({ className, ...props }, ref) => /* @__PURE__ */ jsx(
  ToastPrimitives.Viewport,
  {
    ref,
    className: cn(
      "fixed h-full top-0 z-[100] flex max-h-screen w-full flex-col-reverse p-4 sm:bottom-0 sm:right-0 sm:top-auto sm:flex-col md:max-w-[420px]",
      className
    ),
    ...props
  }
));
ToastViewport.displayName = ToastPrimitives.Viewport.displayName;
const toastVariants = cva(
  "group pointer-events-auto relative flex w-full items-center justify-between space-x-2 overflow-hidden rounded-md border-t-4 p-4 pr-6 shadow-lg transition-all data-[swipe=cancel]:translate-x-0 data-[swipe=end]:translate-x-[var(--radix-toast-swipe-end-x)] data-[swipe=move]:translate-x-[var(--radix-toast-swipe-move-x)] data-[swipe=move]:transition-none data-[state=open]:animate-in data-[state=closed]:animate-out data-[swipe=end]:animate-out data-[state=closed]:fade-out-80 data-[state=closed]:slide-out-to-right-full data-[state=open]:slide-in-from-top-full data-[state=open]:sm:slide-in-from-bottom-full",
  {
    variants: {
      variant: {
        default: "border bg-background text-foreground",
        success: "bg-background group border-[#22C55E]",
        destructive: "destructive bg-background group border-destructive"
      }
    },
    defaultVariants: {
      variant: "default"
    }
  }
);
const Toast = React.forwardRef(({ className, variant, ...props }, ref) => {
  return /* @__PURE__ */ jsx(
    ToastPrimitives.Root,
    {
      ref,
      className: cn(toastVariants({ variant }), className),
      ...props
    }
  );
});
Toast.displayName = ToastPrimitives.Root.displayName;
const ToastAction = React.forwardRef(({ className, ...props }, ref) => /* @__PURE__ */ jsx(
  ToastPrimitives.Action,
  {
    ref,
    className: cn(
      "inline-flex h-8 shrink-0 items-center justify-center rounded-md border bg-transparent px-3 text-sm font-medium transition-colors hover:bg-secondary focus:outline-none focus:ring-1 focus:ring-ring disabled:pointer-events-none disabled:opacity-50 group-[.destructive]:border-muted/40 group-[.destructive]:hover:border-destructive/30 group-[.destructive]:hover:bg-destructive group-[.destructive]:hover:text-destructive-foreground group-[.destructive]:focus:ring-destructive",
      className
    ),
    ...props
  }
));
ToastAction.displayName = ToastPrimitives.Action.displayName;
const ToastClose = React.forwardRef(({ className, ...props }, ref) => /* @__PURE__ */ jsx(
  ToastPrimitives.Close,
  {
    ref,
    className: cn(
      "absolute right-1 top-1 rounded-md p-1 text-foreground/50 hover:text-foreground focus:opacity-100 focus:outline-none focus:ring-1 group-hover:opacity-100  group-[.destructive]:hover:text-red-50 group-[.destructive]:focus:ring-red-400 group-[.destructive]:focus:ring-offset-red-600",
      className
    ),
    "toast-close": "",
    ...props,
    children: /* @__PURE__ */ jsx(X, { className: "h-4 w-4" })
  }
));
ToastClose.displayName = ToastPrimitives.Close.displayName;
const ToastTitle = React.forwardRef(({ className, ...props }, ref) => /* @__PURE__ */ jsx(
  ToastPrimitives.Title,
  {
    ref,
    className: cn("text-sm font-semibold [&+div]:text-xs", className),
    ...props
  }
));
ToastTitle.displayName = ToastPrimitives.Title.displayName;
const ToastDescription = React.forwardRef(({ className, ...props }, ref) => /* @__PURE__ */ jsx(
  ToastPrimitives.Description,
  {
    ref,
    className: cn("text-sm opacity-90", className),
    ...props
  }
));
ToastDescription.displayName = ToastPrimitives.Description.displayName;
function Toaster() {
  const { toasts } = useToast();
  return /* @__PURE__ */ jsxs(ToastProvider, { children: [
    toasts.map(function({
      id,
      title,
      description,
      action,
      icon,
      ...props
    }) {
      return /* @__PURE__ */ jsxs(Toast, { ...props, children: [
        /* @__PURE__ */ jsxs("div", { className: "flex gap-2", children: [
          /* @__PURE__ */ jsx("div", { className: "mt-1", children: icon }),
          /* @__PURE__ */ jsxs("div", { className: "grid gap-1", children: [
            title && /* @__PURE__ */ jsx(ToastTitle, { children: title }),
            description && /* @__PURE__ */ jsx(ToastDescription, { children: description })
          ] }),
          action
        ] }),
        /* @__PURE__ */ jsx(ToastClose, {})
      ] }, id);
    }),
    /* @__PURE__ */ jsx(ToastViewport, {})
  ] });
}
const links = () => [
  { rel: "preconnect", href: "https://fonts.googleapis.com" },
  {
    rel: "preconnect",
    href: "https://fonts.gstatic.com",
    crossOrigin: "anonymous"
  },
  {
    rel: "stylesheet",
    href: "https://fonts.googleapis.com/css2?family=Open+Sans:ital,opsz,wght@0,14..32,100..900;1,14..32,100..900&display=swap"
  }
];
function Layout({ children }) {
  return /* @__PURE__ */ jsxs("html", { lang: "en", children: [
    /* @__PURE__ */ jsxs("head", { children: [
      /* @__PURE__ */ jsx("meta", { charSet: "utf-8" }),
      /* @__PURE__ */ jsx("meta", { name: "viewport", content: "width=device-width, initial-scale=1" }),
      /* @__PURE__ */ jsx("title", { children: "OnePrint" }),
      /* @__PURE__ */ jsx(Meta, {}),
      /* @__PURE__ */ jsx(Links, {})
    ] }),
    /* @__PURE__ */ jsxs("body", { children: [
      children,
      /* @__PURE__ */ jsx(ScrollRestoration, {}),
      /* @__PURE__ */ jsx(Scripts, {}),
      /* @__PURE__ */ jsx(Toaster, {})
    ] })
  ] });
}
function App() {
  return /* @__PURE__ */ jsx(Provider, { store, children: /* @__PURE__ */ jsx(Outlet, {}) });
}
const route0 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  Layout,
  default: App,
  links
}, Symbol.toStringTag, { value: "Module" }));
const authImg = "/assets/auth-img-xMGwVesW.png";
const bg1 = "/assets/bg1-G0AHkeGQ.jpeg";
const bg2 = "/assets/bg2-WSE4bN91.jpeg";
const bg3 = "/assets/bg3-DToVRWpF.jpeg";
const bg4 = "/assets/bg4-B2hxTYhK.jpeg";
const bg5 = "/assets/bg5-D0vehbPI.jpeg";
const product1 = "/assets/product1-BNAff3gy.jpeg";
const product2 = "/assets/product2-C-tY9Hnw.jpeg";
const product3 = "/assets/product3-IgBT5b-X.jpeg";
const product4 = "/assets/product4-C1XsPnou.jpeg";
const product5 = "/assets/product5-CAbwSwrg.jpeg";
const product6 = "/assets/product6-CDZdddmu.jpeg";
const product7 = "/assets/product7-BT-c3d3m.jpeg";
const product8 = "/assets/product8-DSAYakeI.jpeg";
const product9 = "/assets/product9-I-O5Cu9O.jpeg";
const category1 = "/assets/category1-DLnhxDpj.jpg";
const category2 = "/assets/category2-BE9G3ls3.jpg";
const category3 = "/assets/category3-D5z4SAN0.jpg";
const category4 = "/assets/category4-DUiR4XCH.jpg";
const category5 = "/assets/category5-iERaFdCt.jpg";
const category6 = "/assets/category6-CDvSRJ04.jpg";
const category7 = "/assets/category7-C94LGwmB.jpg";
const category8 = "/assets/category8-D1-aEEwx.jpg";
const brand1 = "/assets/brand1-DkA_KlbG.svg";
const brand2 = "/assets/brand2-CXlpkMKH.svg";
const brand3 = "/assets/brand3-DjEdlAsk.svg";
const brand4 = "/assets/brand4-C2RK9Hb8.svg";
const brand5 = "/assets/brand5-CGCNDSW6.svg";
const brand6 = "/assets/brand6-C9xKBTUO.svg";
const brand7 = "/assets/brand7-D68zs0MY.svg";
const brand8 = "/assets/brand8-BT2h9zzH.svg";
const brand9 = "/assets/brand9-C_rF9FZ5.svg";
const brand10 = "/assets/brand10-OMqggiq-.svg";
const brand11 = "/assets/brand11-DEoa_DAO.svg";
const brand12 = "/assets/brand12-w8xPC_Tp.svg";
const logo = "/assets/logo-DyLVSTuq.svg";
const facebook = "data:image/svg+xml,%3csvg%20width='42'%20height='42'%20viewBox='0%200%2042%2042'%20fill='none'%20xmlns='http://www.w3.org/2000/svg'%3e%3crect%20x='0.5'%20y='0.5'%20width='41'%20height='41'%20rx='20.5'%20stroke='%23E2E8F0'/%3e%3cg%20clip-path='url(%23clip0_225_1122)'%3e%3cpath%20d='M32%2021C32%2014.9249%2027.0751%2010%2021%2010C14.9249%2010%2010%2014.9249%2010%2021C10%2026.4903%2014.0225%2031.0412%2019.2812%2031.8664V24.1797H16.4883V21H19.2812V18.5766C19.2812%2015.8197%2020.9235%2014.2969%2023.4361%2014.2969C24.6392%2014.2969%2025.8984%2014.5117%2025.8984%2014.5117V17.2187H24.5114C23.145%2017.2187%2022.7188%2018.0667%2022.7188%2018.9375V21H25.7695L25.2818%2024.1797H22.7188V31.8664C27.9775%2031.0412%2032%2026.4903%2032%2021Z'%20fill='%231877F2'/%3e%3cpath%20d='M25.2818%2024.1797L25.7695%2021H22.7188V18.9375C22.7188%2018.0676%2023.145%2017.2188%2024.5114%2017.2188H25.8984V14.5117C25.8984%2014.5117%2024.6397%2014.2969%2023.4361%2014.2969C20.9235%2014.2969%2019.2812%2015.8197%2019.2812%2018.5766V21H16.4883V24.1797H19.2812V31.8664C20.4202%2032.0445%2021.5798%2032.0445%2022.7188%2031.8664V24.1797H25.2818Z'%20fill='white'/%3e%3c/g%3e%3cdefs%3e%3cclipPath%20id='clip0_225_1122'%3e%3crect%20width='22'%20height='22'%20fill='white'%20transform='translate(10%2010)'/%3e%3c/clipPath%3e%3c/defs%3e%3c/svg%3e";
const google = "data:image/svg+xml,%3csvg%20width='42'%20height='42'%20viewBox='0%200%2042%2042'%20fill='none'%20xmlns='http://www.w3.org/2000/svg'%3e%3crect%20x='0.5'%20y='0.5'%20width='41'%20height='41'%20rx='20.5'%20stroke='%23E2E8F0'/%3e%3cg%20clip-path='url(%23clip0_225_1120)'%3e%3cpath%20d='M31.7857%2021.2534C31.7857%2020.5056%2031.7251%2019.7538%2031.5958%2019.0182H21.2202V23.2541H27.1618C26.9152%2024.6202%2026.123%2025.8288%2024.963%2026.5967V29.3452H28.5078C30.5893%2027.4294%2031.7857%2024.6%2031.7857%2021.2534Z'%20fill='%234285F4'/%3e%3cpath%20d='M21.2201%2032.0008C24.1869%2032.0008%2026.6888%2031.0267%2028.5117%2029.3453L24.967%2026.5968C23.9807%2027.2677%2022.7075%2027.6477%2021.2242%2027.6477C18.3544%2027.6477%2015.9212%2025.7116%2015.0482%2023.1086H11.3903V25.942C13.2576%2029.6565%2017.061%2032.0008%2021.2201%2032.0008Z'%20fill='%2334A853'/%3e%3cpath%20d='M15.0442%2023.1086C14.5834%2021.7424%2014.5834%2020.2631%2015.0442%2018.897V16.0636H11.3903C9.83017%2019.1718%209.83017%2022.8338%2011.3903%2025.942L15.0442%2023.1086Z'%20fill='%23FBBC04'/%3e%3cpath%20d='M21.2201%2014.3539C22.7884%2014.3296%2024.3041%2014.9197%2025.4399%2016.0029L28.5804%2012.8624C26.5918%2010.995%2023.9524%209.96841%2021.2201%2010.0007C17.061%2010.0007%2013.2576%2012.345%2011.3903%2016.0636L15.0441%2018.8969C15.9131%2016.2899%2018.3504%2014.3539%2021.2201%2014.3539Z'%20fill='%23EA4335'/%3e%3c/g%3e%3cdefs%3e%3cclipPath%20id='clip0_225_1120'%3e%3crect%20width='22'%20height='22'%20fill='white'%20transform='translate(10%2010)'/%3e%3c/clipPath%3e%3c/defs%3e%3c/svg%3e";
const apple = "data:image/svg+xml,%3csvg%20width='42'%20height='42'%20viewBox='0%200%2042%2042'%20fill='none'%20xmlns='http://www.w3.org/2000/svg'%3e%3crect%20x='0.5'%20y='0.5'%20width='41'%20height='41'%20rx='20.5'%20stroke='%23E2E8F0'/%3e%3cg%20clip-path='url(%23clip0_225_1124)'%3e%3cpath%20d='M29.976%2027.1449C29.6433%2027.9135%2029.2495%2028.6211%2028.7932%2029.2715C28.1712%2030.1583%2027.662%2030.7721%2027.2695%2031.113C26.6612%2031.6724%2026.0093%2031.959%2025.3113%2031.9753C24.8102%2031.9753%2024.2059%2031.8327%2023.5025%2031.5434C22.7968%2031.2555%2022.1482%2031.113%2021.5552%2031.113C20.9332%2031.113%2020.2662%2031.2555%2019.5527%2031.5434C18.8381%2031.8327%2018.2625%2031.9834%2017.8223%2031.9984C17.153%2032.0269%2016.4858%2031.7322%2015.8199%2031.113C15.3948%2030.7422%2014.8632%2030.1067%2014.2263%2029.2063C13.5429%2028.2449%2012.9811%2027.13%2012.541%2025.8589C12.0697%2024.486%2011.8334%2023.1565%2011.8334%2021.8694C11.8334%2020.3951%2012.152%2019.1235%2012.7901%2018.0578C13.2916%2017.2019%2013.9588%2016.5267%2014.7938%2016.0311C15.6288%2015.5354%2016.531%2015.2828%2017.5027%2015.2666C18.0343%2015.2666%2018.7315%2015.4311%2019.5979%2015.7543C20.4619%2016.0786%2021.0166%2016.243%2021.2598%2016.243C21.4416%2016.243%2022.0579%2016.0507%2023.1026%2015.6674C24.0905%2015.3119%2024.9243%2015.1647%2025.6074%2015.2226C27.4583%2015.372%2028.8489%2016.1017%2029.7737%2017.4162C28.1183%2018.4192%2027.2994%2019.824%2027.3157%2021.6262C27.3307%2023.03%2027.8399%2024.1981%2028.8407%2025.1256C29.2943%2025.5561%2029.8008%2025.8888%2030.3644%2026.1251C30.2422%2026.4795%2030.1132%2026.819%2029.976%2027.1449ZM25.731%2010.4404C25.731%2011.5406%2025.329%2012.5679%2024.5278%2013.5188C23.5609%2014.6492%2022.3914%2015.3024%2021.1232%2015.1993C21.107%2015.0673%2021.0977%2014.9284%2021.0977%2014.7824C21.0977%2013.7261%2021.5575%2012.5958%2022.374%2011.6715C22.7817%2011.2036%2023.3002%2010.8145%2023.9289%2010.5041C24.5563%2010.1983%2025.1497%2010.0292%2025.7079%2010.0002C25.7242%2010.1473%2025.731%2010.2944%2025.731%2010.4404V10.4404Z'%20fill='black'/%3e%3c/g%3e%3cdefs%3e%3cclipPath%20id='clip0_225_1124'%3e%3crect%20width='22'%20height='22'%20fill='white'%20transform='translate(10%2010)'/%3e%3c/clipPath%3e%3c/defs%3e%3c/svg%3e";
const visa = "data:image/svg+xml,%3csvg%20width='66'%20height='50'%20viewBox='0%200%2066%2050'%20fill='none'%20xmlns='http://www.w3.org/2000/svg'%3e%3cg%20filter='url(%23filter0_dd_252_1151)'%3e%3crect%20x='10'%20y='5.48486'%20width='46.6143'%20height='30.0304'%20rx='5.37857'%20fill='white'/%3e%3crect%20x='9.88795'%20y='5.37281'%20width='46.8384'%20height='30.2545'%20rx='5.49062'%20stroke='%23D6DCE5'%20stroke-width='0.224107'/%3e%3cpath%20d='M33.0222%2015.6059L30.848%2025.7694H28.2189L30.3935%2015.6059H33.0222ZM44.0839%2022.1685L45.4681%2018.3517L46.2646%2022.1685H44.0839ZM47.0173%2025.7694H49.4492L47.3271%2015.6059H45.0824C44.578%2015.6059%2044.1524%2015.8992%2043.963%2016.3515L40.0185%2025.7694H42.7795L43.3276%2024.2517H46.7008L47.0173%2025.7694ZM40.1555%2022.4509C40.1668%2019.7684%2036.4459%2019.6208%2036.4716%2018.4225C36.4794%2018.0574%2036.8271%2017.6701%2037.5866%2017.5707C37.9639%2017.5215%2039.0013%2017.4838%2040.1793%2018.0259L40.6406%2015.8706C40.0079%2015.6409%2039.1937%2015.4199%2038.1811%2015.4199C35.5825%2015.4199%2033.7534%2016.8015%2033.738%2018.7799C33.7216%2020.2429%2035.0437%2021.0593%2036.0399%2021.5457C37.0642%2022.044%2037.4084%2022.3631%2037.4039%2022.8086C37.3971%2023.4909%2036.5868%2023.7914%2035.8308%2023.8033C34.5087%2023.8242%2033.7415%2023.4465%2033.13%2023.1615L32.6536%2025.3885C33.2677%2025.6706%2034.4016%2025.9158%2035.578%2025.9283C38.3396%2025.9283%2040.1465%2024.5637%2040.1555%2022.4509ZM29.2653%2015.6059L25.0056%2025.7694H22.226L20.1299%2017.6582C20.0025%2017.1586%2019.8918%2016.9759%2019.5049%2016.7652C18.8734%2016.4226%2017.8295%2016.1006%2016.9111%2015.9012L16.9739%2015.6059H21.4475C22.0175%2015.6059%2022.5306%2015.9854%2022.6599%2016.642L23.7671%2022.5236L26.503%2015.6059H29.2653Z'%20fill='%231434CB'/%3e%3c/g%3e%3cdefs%3e%3cfilter%20id='filter0_dd_252_1151'%20x='0.811593'%20y='0.778599'%20width='64.9911'%20height='48.4071'%20filterUnits='userSpaceOnUse'%20color-interpolation-filters='sRGB'%3e%3cfeFlood%20flood-opacity='0'%20result='BackgroundImageFix'/%3e%3cfeColorMatrix%20in='SourceAlpha'%20type='matrix'%20values='0%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%20127%200'%20result='hardAlpha'/%3e%3cfeOffset%20dy='0.448214'/%3e%3cfeGaussianBlur%20stdDeviation='2.24107'/%3e%3cfeColorMatrix%20type='matrix'%20values='0%200%200%200%200.717647%200%200%200%200%200.717647%200%200%200%200%200.717647%200%200%200%200.08%200'/%3e%3cfeBlend%20mode='normal'%20in2='BackgroundImageFix'%20result='effect1_dropShadow_252_1151'/%3e%3cfeColorMatrix%20in='SourceAlpha'%20type='matrix'%20values='0%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%20127%200'%20result='hardAlpha'/%3e%3cfeOffset%20dy='4.48214'/%3e%3cfeGaussianBlur%20stdDeviation='4.48214'/%3e%3cfeColorMatrix%20type='matrix'%20values='0%200%200%200%200.717647%200%200%200%200%200.717647%200%200%200%200%200.717647%200%200%200%200.08%200'/%3e%3cfeBlend%20mode='normal'%20in2='effect1_dropShadow_252_1151'%20result='effect2_dropShadow_252_1151'/%3e%3cfeBlend%20mode='normal'%20in='SourceGraphic'%20in2='effect2_dropShadow_252_1151'%20result='shape'/%3e%3c/filter%3e%3c/defs%3e%3c/svg%3e";
const paypal = "/assets/paypal-B6KRm9ok.svg";
const googlepay = "/assets/googlepay-C6eWBuah.svg";
const applepay = "data:image/svg+xml,%3csvg%20width='66'%20height='50'%20viewBox='0%200%2066%2050'%20fill='none'%20xmlns='http://www.w3.org/2000/svg'%3e%3cg%20filter='url(%23filter0_dd_252_1157)'%3e%3crect%20x='9.84277'%20y='5.48486'%20width='46.6143'%20height='30.0304'%20rx='5.37857'%20fill='white'/%3e%3crect%20x='9.73072'%20y='5.37281'%20width='46.8384'%20height='30.2545'%20rx='5.49062'%20stroke='%23D6DCE5'%20stroke-width='0.224107'/%3e%3cpath%20d='M24.92%2016.5666C25.231%2016.1776%2025.4421%2015.6553%2025.3864%2015.1216C24.9311%2015.1442%2024.3756%2015.4219%2024.054%2015.8112C23.7652%2016.1446%2023.5096%2016.6887%2023.5762%2017.2001C24.0873%2017.2444%2024.5978%2016.9446%2024.92%2016.5666Z'%20fill='%23000008'/%3e%3cpath%20d='M25.3806%2017.2999C24.6385%2017.2557%2024.0074%2017.7211%2023.653%2017.7211C23.2984%2017.7211%2022.7557%2017.3222%2022.1687%2017.333C21.4047%2017.3442%2020.6957%2017.7762%2020.308%2018.4632C19.5106%2019.8377%2020.0976%2021.8765%2020.8731%2022.9959C21.2497%2023.5497%2021.7035%2024.1595%2022.3015%2024.1376C22.8666%2024.1154%2023.088%2023.7717%2023.7749%2023.7717C24.4612%2023.7717%2024.6607%2024.1376%2025.2588%2024.1265C25.879%2024.1154%2026.2668%2023.5724%2026.6434%2023.0181C27.0754%2022.3867%2027.2523%2021.7772%2027.2634%2021.7437C27.2523%2021.7326%2026.0674%2021.278%2026.0564%2019.915C26.0452%2018.7737%2026.9867%2018.2309%2027.031%2018.1972C26.4993%2017.4109%2025.6686%2017.3222%2025.3806%2017.2999Z'%20fill='%23000008'/%3e%3cpath%20d='M31.8427%2015.7554C33.4557%2015.7554%2034.579%2016.8673%2034.579%2018.4861C34.579%2020.1107%2033.4326%2021.2284%2031.8022%2021.2284H30.0163V24.0686H28.7259V15.7554L31.8427%2015.7554ZM30.0163%2020.1453H31.4969C32.6203%2020.1453%2033.2597%2019.5404%2033.2597%2018.4919C33.2597%2017.4435%2032.6203%2016.8443%2031.5026%2016.8443H30.0163V20.1453Z'%20fill='%23000008'/%3e%3cpath%20d='M34.9161%2022.3461C34.9161%2021.2859%2035.7284%2020.635%2037.1688%2020.5543L38.8278%2020.4564V19.9898C38.8278%2019.3157%2038.3727%2018.9125%2037.6124%2018.9125C36.8921%2018.9125%2036.4427%2019.2581%2036.3333%2019.7997H35.1581C35.2272%2018.705%2036.1604%2017.8985%2037.6584%2017.8985C39.1274%2017.8985%2040.0664%2018.6762%2040.0664%2019.8918V24.0686H38.8738V23.0719H38.8451C38.4938%2023.746%2037.7275%2024.1723%2036.9325%2024.1723C35.7457%2024.1723%2034.9161%2023.4348%2034.9161%2022.3461ZM38.8278%2021.7988V21.3206L37.3357%2021.4127C36.5925%2021.4646%2036.172%2021.793%2036.172%2022.3115C36.172%2022.8414%2036.6098%2023.1871%2037.2781%2023.1871C38.148%2023.1871%2038.8278%2022.588%2038.8278%2021.7988Z'%20fill='%23000008'/%3e%3cpath%20d='M41.1923%2026.2981V25.2899C41.2843%2025.3129%2041.4916%2025.3129%2041.5954%2025.3129C42.1715%2025.3129%2042.4826%2025.071%2042.6727%2024.4488C42.6727%2024.4373%2042.7822%2024.0801%2042.7822%2024.0744L40.5931%2018.0079H41.941L43.4736%2022.9395H43.4965L45.0291%2018.0079H46.3426L44.0726%2024.3854C43.5543%2025.8545%2042.9551%2026.3269%2041.6992%2026.3269C41.5954%2026.3269%2041.2843%2026.3154%2041.1923%2026.2981Z'%20fill='%23000008'/%3e%3c/g%3e%3cdefs%3e%3cfilter%20id='filter0_dd_252_1157'%20x='0.654366'%20y='0.778599'%20width='64.9911'%20height='48.4071'%20filterUnits='userSpaceOnUse'%20color-interpolation-filters='sRGB'%3e%3cfeFlood%20flood-opacity='0'%20result='BackgroundImageFix'/%3e%3cfeColorMatrix%20in='SourceAlpha'%20type='matrix'%20values='0%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%20127%200'%20result='hardAlpha'/%3e%3cfeOffset%20dy='0.448214'/%3e%3cfeGaussianBlur%20stdDeviation='2.24107'/%3e%3cfeColorMatrix%20type='matrix'%20values='0%200%200%200%200.717647%200%200%200%200%200.717647%200%200%200%200%200.717647%200%200%200%200.08%200'/%3e%3cfeBlend%20mode='normal'%20in2='BackgroundImageFix'%20result='effect1_dropShadow_252_1157'/%3e%3cfeColorMatrix%20in='SourceAlpha'%20type='matrix'%20values='0%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%20127%200'%20result='hardAlpha'/%3e%3cfeOffset%20dy='4.48214'/%3e%3cfeGaussianBlur%20stdDeviation='4.48214'/%3e%3cfeColorMatrix%20type='matrix'%20values='0%200%200%200%200.717647%200%200%200%200%200.717647%200%200%200%200%200.717647%200%200%200%200.08%200'/%3e%3cfeBlend%20mode='normal'%20in2='effect1_dropShadow_252_1157'%20result='effect2_dropShadow_252_1157'/%3e%3cfeBlend%20mode='normal'%20in='SourceGraphic'%20in2='effect2_dropShadow_252_1157'%20result='shape'/%3e%3c/filter%3e%3c/defs%3e%3c/svg%3e";
const mastercard = "data:image/svg+xml,%3csvg%20width='66'%20height='50'%20viewBox='0%200%2066%2050'%20fill='none'%20xmlns='http://www.w3.org/2000/svg'%3e%3cg%20filter='url(%23filter0_dd_252_1153)'%3e%3crect%20x='9.61426'%20y='5.48486'%20width='46.6143'%20height='30.0304'%20rx='5.37857'%20fill='white'/%3e%3crect%20x='9.5022'%20y='5.37281'%20width='46.8384'%20height='30.2545'%20rx='5.49062'%20stroke='%23D6DCE5'%20stroke-width='0.224107'/%3e%3cpath%20d='M36.27%2014.3333H29.4068V26.6668H36.27V14.3333Z'%20fill='%23FF5F00'/%3e%3cpath%20d='M29.8426%2020.5C29.8415%2019.3122%2030.1107%2018.1397%2030.6297%2017.0713C31.1487%2016.0029%2031.904%2015.0666%2032.8384%2014.3332C31.6813%2013.4237%2030.2916%2012.8581%2028.8283%2012.701C27.3649%2012.544%2025.8869%2012.8018%2024.5631%2013.445C23.2393%2014.0883%2022.1232%2015.091%2021.3424%2016.3386C20.5615%2017.5861%2020.1475%2019.0282%2020.1475%2020.5C20.1475%2021.9718%2020.5615%2023.4139%2021.3424%2024.6614C22.1232%2025.909%2023.2393%2026.9117%2024.5631%2027.555C25.8869%2028.1982%2027.3649%2028.456%2028.8283%2028.299C30.2916%2028.1419%2031.6813%2027.5763%2032.8384%2026.6668C31.904%2025.9334%2031.1487%2024.9971%2030.6297%2023.9287C30.1107%2022.8603%2029.8415%2021.6878%2029.8426%2020.5Z'%20fill='%23EB001B'/%3e%3cpath%20d='M45.529%2020.5C45.529%2021.9718%2045.115%2023.4138%2044.3342%2024.6614C43.5534%2025.909%2042.4374%2026.9117%2041.1136%2027.5549C39.7899%2028.1982%2038.3118%2028.456%2036.8485%2028.299C35.3851%2028.1419%2033.9955%2027.5763%2032.8384%2026.6668C33.772%2025.9327%2034.5267%2024.9962%2035.0457%2023.928C35.5646%2022.8597%2035.8343%2021.6876%2035.8343%2020.5C35.8343%2019.3124%2035.5646%2018.1403%2035.0457%2017.072C34.5267%2016.0038%2033.772%2015.0673%2032.8384%2014.3332C33.9955%2013.4237%2035.3851%2012.8581%2036.8485%2012.701C38.3118%2012.544%2039.7899%2012.8018%2041.1136%2013.4451C42.4374%2014.0883%2043.5534%2015.091%2044.3342%2016.3386C45.115%2017.5862%2045.529%2019.0282%2045.529%2020.5Z'%20fill='%23F79E1B'/%3e%3cpath%20d='M44.7807%2025.3604V25.1079H44.8825V25.0565H44.6232V25.1079H44.725V25.3604H44.7807ZM45.2841%2025.3604V25.056H45.2046L45.1132%2025.2654L45.0217%2025.056H44.9422V25.3604H44.9983V25.1308L45.0841%2025.3288H45.1423L45.228%2025.1303V25.3604H45.2841Z'%20fill='%23F79E1B'/%3e%3c/g%3e%3cdefs%3e%3cfilter%20id='filter0_dd_252_1153'%20x='0.425851'%20y='0.778599'%20width='64.9911'%20height='48.4071'%20filterUnits='userSpaceOnUse'%20color-interpolation-filters='sRGB'%3e%3cfeFlood%20flood-opacity='0'%20result='BackgroundImageFix'/%3e%3cfeColorMatrix%20in='SourceAlpha'%20type='matrix'%20values='0%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%20127%200'%20result='hardAlpha'/%3e%3cfeOffset%20dy='0.448214'/%3e%3cfeGaussianBlur%20stdDeviation='2.24107'/%3e%3cfeColorMatrix%20type='matrix'%20values='0%200%200%200%200.717647%200%200%200%200%200.717647%200%200%200%200%200.717647%200%200%200%200.08%200'/%3e%3cfeBlend%20mode='normal'%20in2='BackgroundImageFix'%20result='effect1_dropShadow_252_1153'/%3e%3cfeColorMatrix%20in='SourceAlpha'%20type='matrix'%20values='0%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%200%20127%200'%20result='hardAlpha'/%3e%3cfeOffset%20dy='4.48214'/%3e%3cfeGaussianBlur%20stdDeviation='4.48214'/%3e%3cfeColorMatrix%20type='matrix'%20values='0%200%200%200%200.717647%200%200%200%200%200.717647%200%200%200%200%200.717647%200%200%200%200.08%200'/%3e%3cfeBlend%20mode='normal'%20in2='effect1_dropShadow_252_1153'%20result='effect2_dropShadow_252_1153'/%3e%3cfeBlend%20mode='normal'%20in='SourceGraphic'%20in2='effect2_dropShadow_252_1153'%20result='shape'/%3e%3c/filter%3e%3c/defs%3e%3c/svg%3e";
const amazon = "/assets/amazon-CqU2bwsE.svg";
const comedyCentral = "/assets/comedy-central-m-Mf_b8R.svg";
const makeWish = "/assets/make-wish-WUdUS7o1.svg";
const nascar = "/assets/nascar-JELxeAee.svg";
const oracle = "/assets/oracle-D-gva5Lr.svg";
const images = {
  authImg,
  bg1,
  bg2,
  bg3,
  bg4,
  bg5,
  product1,
  product2,
  product3,
  product4,
  product5,
  product6,
  product7,
  product8,
  product9,
  brand1,
  brand2,
  brand3,
  brand4,
  brand5,
  brand6,
  brand7,
  brand8,
  brand9,
  brand10,
  brand11,
  brand12,
  category1,
  category2,
  category3,
  category4,
  category5,
  category6,
  category7,
  category8
};
const svgs = {
  logo,
  facebook,
  google,
  apple,
  visa,
  paypal,
  googlepay,
  applepay,
  mastercard,
  amazon,
  comedyCentral,
  makeWish,
  nascar,
  oracle
};
function useLocalStorage(key, initialValue) {
  const [storedValue, setStoredValue] = useState(() => {
    if (typeof window === "undefined") {
      return initialValue;
    }
    try {
      const item = window.localStorage.getItem(key);
      return item ? JSON.parse(item) : initialValue;
    } catch (error) {
      console.error("Error reading localStorage during initialization:", error);
      return initialValue;
    }
  });
  useEffect(() => {
    if (typeof window !== "undefined") {
      try {
        window.localStorage.setItem(key, JSON.stringify(storedValue));
      } catch (error) {
        console.error("Error setting localStorage key:", key, error);
      }
    }
  }, [key, storedValue]);
  const getStoredValue = () => {
    if (typeof window === "undefined") {
      console.warn("getStoredValue was called on the server.");
      return initialValue;
    }
    try {
      const item = window.localStorage.getItem(key);
      return item ? JSON.parse(item) : initialValue;
    } catch (error) {
      console.error("Error reading localStorage key:", key, error);
      return initialValue;
    }
  };
  const setValue = (value) => {
    try {
      const valueToStore = value instanceof Function ? value(storedValue) : value;
      setStoredValue(valueToStore);
    } catch (error) {
      console.error("Error updating state for localStorage key:", key, error);
    }
  };
  return [storedValue, setValue, getStoredValue];
}
const useAuth = () => {
  const user = useSelector(selectCurrentUser);
  const [storedUser, setStoredUser] = useLocalStorage("authUser", {
    access: "",
    refresh: "",
    user: { pk: 0, username: "", email: "", first_name: "", last_name: "" }
  });
  useEffect(() => {
    if (user) {
      setStoredUser(user);
    }
  }, [user, setStoredUser]);
  return useMemo(() => {
    return { user: storedUser || user };
  }, [storedUser, user]);
};
const AuthLayout = ({ children, title, subtitle }) => {
  const navigate = useNavigate();
  const { user } = useAuth();
  useEffect(() => {
    if (user == null ? void 0 : user.access) {
      navigate("/");
    }
  }, [user, navigate]);
  const advantage = [
    "Design, customize, and shop seamlessly",
    "Access your saved designs",
    "Track your orders"
  ];
  return /* @__PURE__ */ jsxs("div", { className: "flex h-screen", children: [
    /* @__PURE__ */ jsxs("div", { className: "flex flex-col justify-between items-start bg-[url('/app/assets/auth/grid-background.svg')] bg-contain w-full h-full bg-[#101212] py-12 px-3", children: [
      /* @__PURE__ */ jsxs("div", { className: "flex flex-col gap-5 max-w-xl mx-auto", children: [
        /* @__PURE__ */ jsx("p", { className: "text-white text-5xl leading-snug font-semibold", children: title }),
        /* @__PURE__ */ jsx("p", { className: "text-white text-xl", children: subtitle })
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "flex flex-col gap-5 max-w-xl mx-auto text-white w-full", children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: images.authImg,
            alt: "Profile Images",
            className: "h-12 w-fit"
          }
        ),
        /* @__PURE__ */ jsx("p", { className: "text-lg", children: "Join 10,000+ People Creating and Personalizing Their Products" }),
        advantage.map((item, index) => /* @__PURE__ */ jsxs("div", { className: "flex gap-2 items-center text-lg", children: [
          /* @__PURE__ */ jsx("div", { className: "text-primary border-2 border-primary w-7 h-7 rounded-full pt-1 px-0.5", children: /* @__PURE__ */ jsx(Check, { size: 18 }) }),
          item
        ] }, index))
      ] })
    ] }),
    /* @__PURE__ */ jsx("div", { className: "w-full h-full overflow-y-auto py-5", children: /* @__PURE__ */ jsxs("div", { className: "flex flex-col gap-y-24 max-w-xl h-full mx-auto px-3", children: [
      /* @__PURE__ */ jsx("div", { children: /* @__PURE__ */ jsx("img", { src: svgs.logo, alt: "Profile Images", className: "h-10 w-fit" }) }),
      /* @__PURE__ */ jsx("div", { className: "h-full", children })
    ] }) })
  ] });
};
const Heading = ({ title, subtitle }) => {
  return /* @__PURE__ */ jsxs("div", { className: "flex flex-col gap-2.5", children: [
    /* @__PURE__ */ jsx("h2", { className: "text-3xl font-semibold", children: title }),
    /* @__PURE__ */ jsx("h4", { className: "text-lg text-slate-500", children: subtitle })
  ] });
};
const TextInput = ({
  register,
  fieldName,
  fieldLabel,
  type,
  moreCss = "border-stroke",
  required,
  isValidated,
  validateFn,
  handleKeyDown,
  onChangeFn,
  defaultValue,
  placeholder = true,
  isDisabled,
  tooltip
}) => {
  const [password, setPassword] = useState("password");
  const handleChange = (e) => {
    if (onChangeFn) {
      onChangeFn(e.target.value);
    }
  };
  const togglePasswordVisibility = () => {
    setPassword((prev) => prev === "password" ? "text" : "password");
  };
  return /* @__PURE__ */ jsxs("div", { className: "relative w-full", children: [
    /* @__PURE__ */ jsx(
      "input",
      {
        id: fieldName,
        defaultValue,
        className: `
          ${type !== "email" && "capitalize"} w-full rounded-md border-2 bg-transparent py-2.5 px-3 text-black outline-none transition focus:border-primary active:border-primary disabled:cursor-default disabled:bg-white
          ${moreCss}`,
        type: type === "password" ? password : type,
        placeholder: placeholder ? typeof placeholder === "string" ? placeholder : `Enter ${(fieldLabel == null ? void 0 : fieldLabel.toLowerCase()) || startCase(fieldName).toLowerCase()}...` : "",
        ...register(fieldName, {
          required: required ? "This field is required." : false,
          validate: isValidated ? validateFn : void 0,
          onChange: handleChange
        }),
        onKeyDown: handleKeyDown,
        disabled: isDisabled,
        autoComplete: "on"
      }
    ),
    type === "password" && /* @__PURE__ */ jsx(
      "span",
      {
        className: "absolute right-4 bottom-3 text-2xl hover:cursor-pointer",
        onClick: togglePasswordVisibility,
        as: "button",
        children: password === "password" ? /* @__PURE__ */ jsx(Eye, {}) : /* @__PURE__ */ jsx(EyeClosed, {})
      }
    ),
    tooltip && /* @__PURE__ */ jsx("span", { className: "tooltip", children: tooltip })
  ] });
};
const TextArea = ({
  register,
  fieldName,
  fieldLabel,
  required,
  isValidated,
  validateFn,
  handleKeyDown,
  defaultValue,
  placeholder = true,
  isDisabled
}) => {
  return /* @__PURE__ */ jsx(
    "textarea",
    {
      id: fieldName,
      rows: 4,
      defaultValue,
      className: "w-full rounded border-2 bg-transparent py-2.5 px-3 text-black outline-none transition focus:border-primary active:border-primary disabled:cursor-default disabled:bg-whiter dark:border-form-strokedark dark:bg-form-input dark:text-white dark:focus:border-primary",
      placeholder: placeholder ? typeof placeholder === "string" ? placeholder : `Enter ${(fieldLabel == null ? void 0 : fieldLabel.toLowerCase()) || startCase(fieldName).toLowerCase()}...` : "",
      ...register(fieldName, {
        required: required ? "This field is required." : false,
        validate: isValidated ? validateFn : void 0
      }),
      onKeyDown: handleKeyDown,
      autoComplete: "on",
      disabled: isDisabled
    }
  );
};
const PhoneNumber = ({
  control,
  fieldName,
  required = false,
  maxLength,
  isDisabled = false,
  defaultValue,
  moreCss
}) => {
  return /* @__PURE__ */ jsx(
    "div",
    {
      className: `w-full rounded border-2 border-stroke bg-transparent px-3 py-2.5 text-black outline-none transition focus:border-primary active:border-primary disabled:cursor-default disabled:bg-whiter dark:border-form-strokedark dark:bg-form-input dark:text-white dark:focus:border-primary ${moreCss}`,
      children: /* @__PURE__ */ jsx(
        PhoneInput,
        {
          name: fieldName || "phoneNumber",
          control,
          rules: {
            required: required ? "This field is required." : false,
            maxLength: {
              value: maxLength,
              message: "Invalid phone number"
            },
            minLength: {
              value: maxLength,
              message: "Invalid phone number"
            }
          },
          international: true,
          defaultValue,
          defaultCountry: "NG",
          countryCallingCodeEditable: false,
          disabled: isDisabled
        }
      )
    }
  );
};
const SelectInput = ({
  optionsData,
  getOptionLabel,
  getOptionValue,
  fieldName,
  fieldLabel,
  placeholder = true,
  control,
  required,
  isDisabled,
  validateFn,
  onChangeFn,
  defaultValue,
  moreCss = "border-stroke"
}) => {
  const formatOptions = (data) => {
    return data.map((item) => {
      if (typeof item === "string") {
        return { value: item, label: item };
      } else {
        return item;
      }
    });
  };
  const options = formatOptions(optionsData);
  const defaultOption = options.find((option) => option.value === defaultValue) || null;
  return /* @__PURE__ */ jsx(
    Controller,
    {
      name: fieldName,
      control,
      defaultValue: defaultOption,
      rules: {
        required: required ? "This field is required." : false,
        validate: validateFn
      },
      render: ({ field }) => /* @__PURE__ */ jsx(
        Select,
        {
          ...field,
          inputId: fieldName,
          options,
          getOptionLabel,
          getOptionValue,
          value: field.value || defaultOption,
          placeholder: placeholder ? typeof placeholder === "string" ? placeholder : `Select ${(fieldLabel == null ? void 0 : fieldLabel.toLowerCase()) || startCase(fieldName)}...` : void 0,
          onChange: (value) => {
            field.onChange(value);
            onChangeFn ? onChangeFn(value) : void 0;
          },
          isSearchable: true,
          isClearable: true,
          closeMenuOnSelect: true,
          blurInputOnSelect: true,
          className: `${moreCss} rounded border-2 bg-transparent py-2.5 px-3 text-black focus:border-primary active:border-primary disabled:cursor-default disabled:bg-whiter dark:border-form-strokedark dark:bg-form-input dark:text-white dark:focus:border-primary`,
          classNamePrefix: "",
          theme: (theme) => ({
            ...theme,
            colors: {
              ...theme.colors,
              primary25: "#eeffee",
              primary: "#fff"
            }
          }),
          styles: formatReactSelectStyles,
          isDisabled
        }
      )
    }
  );
};
const RadioInput = ({
  fieldName,
  register,
  required = false,
  options,
  withDescription = false,
  defaultChecked = "",
  onChangeFn
}) => {
  return /* @__PURE__ */ jsx("div", { className: "grid grid-cols-2 gap-2 justify-start items-center w-fit", children: options.map((option, index) => {
    const value = typeof option === "object" ? option.value : option;
    const label = typeof option === "object" ? option.label : option;
    const description = typeof option === "object" && withDescription ? option.description : null;
    return /* @__PURE__ */ jsxs(
      "div",
      {
        className: `flex justify-start items-center gap-1.5 rounded-full w-fit border-2 border-slate-200 py-1 px-3 ${typeof option === "object" ? "" : ""}`,
        children: [
          /* @__PURE__ */ jsx(
            "input",
            {
              type: "radio",
              id: index,
              value,
              defaultChecked: value === defaultChecked,
              ...register(fieldName, {
                required: required ? "This field is required." : false
              }),
              onClick: () => onChangeFn && onChangeFn(value, index)
            }
          ),
          /* @__PURE__ */ jsxs("label", { htmlFor: label, children: [
            startCase(label),
            " ",
            /* @__PURE__ */ jsx("p", { children: description })
          ] })
        ]
      },
      index
    );
  }) });
};
const CheckBox = ({
  fieldName,
  fieldLabel,
  jsxFieldLabel,
  register,
  required,
  defaultChecked,
  onChangeFn
}) => {
  return /* @__PURE__ */ jsxs("div", { className: "", children: [
    /* @__PURE__ */ jsx(
      "input",
      {
        id: fieldName,
        type: "checkbox",
        defaultChecked,
        ...register(fieldName, {
          required: required ? "This field is required." : false
        }),
        onChange: ({ target: { checked } }) => onChangeFn && onChangeFn(checked)
      }
    ),
    fieldLabel || jsxFieldLabel && /* @__PURE__ */ jsx("label", { htmlFor: fieldName, children: jsxFieldLabel ? jsxFieldLabel : fieldLabel })
  ] });
};
const StarRating = ({
  control,
  fieldName,
  required,
  defaultValue
}) => {
  return /* @__PURE__ */ jsx(
    Controller,
    {
      name: fieldName,
      control,
      defaultValue,
      rules: { required: required ? "This field is required" : false },
      render: ({ field: { onChange, value } }) => /* @__PURE__ */ jsx(
        StarRatings,
        {
          name: fieldName,
          rating: value ? value : defaultValue,
          starRatedColor: "gold",
          starHoverColor: "gold",
          changeRating: onChange,
          numberOfStars: 5,
          starDimension: "25px",
          starSpacing: "2px"
        }
      )
    }
  );
};
const getFieldType = ({
  register,
  control,
  fieldName,
  fieldLabel,
  type,
  required,
  isValidated,
  validateFn,
  onChangeFn,
  handleKeyDown,
  defaultValue,
  placeholder,
  isDisabled,
  maxLength,
  optionsData = [],
  radioOptions = [],
  withDescription,
  defaultChecked,
  tooltip,
  jsxFieldLabel,
  moreCss
}) => {
  let content = null;
  switch (type) {
    case "text":
    case "number":
    case "date":
    case "password":
    case "email":
      content = /* @__PURE__ */ jsx(
        TextInput,
        {
          type,
          fieldName,
          fieldLabel,
          register,
          required,
          isValidated,
          validateFn,
          handleKeyDown,
          defaultValue,
          placeholder,
          isDisabled,
          tooltip,
          onChangeFn,
          moreCss
        }
      );
      break;
    case "textarea":
      content = /* @__PURE__ */ jsx(
        TextArea,
        {
          register,
          fieldName,
          fieldLabel,
          required,
          isValidated,
          validateFn,
          handleKeyDown,
          defaultValue,
          isDisabled
        }
      );
      break;
    case "phone":
      content = /* @__PURE__ */ jsx(
        PhoneNumber,
        {
          moreCss,
          control,
          fieldName,
          maxLength,
          required,
          isDisabled,
          defaultValue
        }
      );
      break;
    case "select":
      content = /* @__PURE__ */ jsx(
        SelectInput,
        {
          control,
          fieldName,
          fieldLabel,
          defaultValue,
          required,
          optionsData,
          isDisabled,
          onChangeFn,
          moreCss
        }
      );
      break;
    case "star-rating":
      content = /* @__PURE__ */ jsx(
        StarRating,
        {
          fieldName,
          required,
          control,
          defaultValue
        }
      );
      break;
    case "radio-options":
      content = /* @__PURE__ */ jsx(
        RadioInput,
        {
          register,
          fieldName,
          required,
          options: radioOptions,
          withDescription,
          defaultChecked
        }
      );
      break;
    case "checkbox":
      content = /* @__PURE__ */ jsx(
        CheckBox,
        {
          register,
          fieldName,
          fieldLabel,
          required,
          defaultChecked,
          jsxFieldLabel
        }
      );
      break;
  }
  return { content, type };
};
const dynamicFormFieldError = ({
  dynamic,
  errors
}) => {
  var _a, _b, _c, _d, _e, _f, _g, _h, _i, _j, _k, _l, _m, _n, _o, _p, _q, _r, _s, _t, _u;
  let errorMessage = "";
  if (dynamic && errors) {
    if (dynamic.grandParent) {
      if (dynamic.grandParentType === "array" && typeof dynamic.parentIndex === "number") {
        if (dynamic.type === "array") {
          errorMessage = (_e = (_d = (_c = (_b = (_a = errors[dynamic.grandParent]) == null ? void 0 : _a[dynamic.parentIndex]) == null ? void 0 : _b[dynamic.parent]) == null ? void 0 : _c[dynamic.index]) == null ? void 0 : _d[dynamic.fieldName]) == null ? void 0 : _e.message;
        } else {
          errorMessage = (_i = (_h = (_g = (_f = errors[dynamic.grandParent]) == null ? void 0 : _f[dynamic.parentIndex]) == null ? void 0 : _g[dynamic.parent]) == null ? void 0 : _h[dynamic.fieldName]) == null ? void 0 : _i.message;
        }
      } else if (dynamic.grandParentType === "object") {
        if (dynamic.type === "array") {
          errorMessage = (_m = (_l = (_k = (_j = errors[dynamic.grandParent]) == null ? void 0 : _j[dynamic.parent]) == null ? void 0 : _k[dynamic.index]) == null ? void 0 : _l[dynamic.fieldName]) == null ? void 0 : _m.message;
        } else {
          errorMessage = (_p = (_o = (_n = errors[dynamic.grandParent]) == null ? void 0 : _n[dynamic.parent]) == null ? void 0 : _o[dynamic.fieldName]) == null ? void 0 : _p.message;
        }
      }
    } else {
      if (dynamic.type === "array") {
        errorMessage = (_s = (_r = (_q = errors[dynamic.parent]) == null ? void 0 : _q[dynamic.index]) == null ? void 0 : _r[dynamic.fieldName]) == null ? void 0 : _s.message;
      } else {
        errorMessage = (_u = (_t = errors[dynamic.parent]) == null ? void 0 : _t[dynamic.fieldName]) == null ? void 0 : _u.message;
      }
    }
    return errorMessage;
  } else {
    return null;
  }
};
const formatReactSelectStyles = {
  control: (baseStyles, state) => ({
    ...baseStyles,
    border: "none",
    minHeight: "fit-content",
    backgroundColor: state.isDisabled ? "transparent" : "transparent"
  }),
  valueContainer: (baseStyles) => ({
    ...baseStyles,
    padding: 0,
    border: "none",
    outline: "none"
  }),
  input: (baseStyles) => ({
    ...baseStyles,
    margin: 0,
    padding: 0,
    border: "none",
    outline: "none"
  }),
  placeholder: (baseStyles) => ({
    ...baseStyles,
    color: "#ced6de",
    fontFamily: "Open Sans"
  }),
  menu: (baseStyles) => ({
    ...baseStyles,
    marginTop: "0.5rem",
    marginLeft: "-0.75rem",
    color: "#414141"
  }),
  option: (baseStyles, state) => ({
    ...baseStyles,
    cursor: "pointer",
    backgroundColor: state.isSelected ? "#eef6f0" : "white",
    color: state.isSelected ? "#1b7232" : "#414141"
  }),
  indicatorSeparator: () => ({
    display: "none"
  }),
  dropdownIndicator: () => ({
    position: "relative",
    right: "0.2rem",
    bottom: "0.1rem",
    color: "#ced6de",
    cursor: "pointer"
  }),
  clearIndicator: () => ({
    position: "relative",
    right: "0.5rem",
    bottom: "0.1rem",
    color: "red",
    cursor: "pointer"
  })
};
const FormElement = (props) => {
  const { fieldName, fieldLabel, withLabel = true, required, dynamic } = props;
  const {
    register,
    control,
    formState: { errors }
  } = useFormContext();
  const dynamicError = dynamicFormFieldError({ dynamic, errors });
  const { content, type } = getFieldType({ register, control, ...props });
  return /* @__PURE__ */ jsxs("div", { className: "w-full", children: [
    type === "checkbox" ? /* @__PURE__ */ jsx(Fragment, { children: content }) : /* @__PURE__ */ jsxs("div", { className: "flex flex-col gap-2 w-full", children: [
      /* @__PURE__ */ jsxs(
        "label",
        {
          htmlFor: fieldName,
          className: "flex gap-0.5 items-center font-semibold",
          children: [
            withLabel && (fieldLabel || startCase(fieldName)),
            withLabel && required && /* @__PURE__ */ jsx("span", { className: "text-red-500", children: "*" })
          ]
        }
      ),
      content
    ] }),
    dynamicError ? /* @__PURE__ */ jsx("p", { className: "text-sm text-red-500 mt-1", children: dynamicError }) : errors[fieldName] && /* @__PURE__ */ jsx(
      ErrorMessage,
      {
        errors,
        name: fieldName,
        render: ({ message }) => /* @__PURE__ */ jsx("p", { className: "text-sm text-red-500 mt-1", children: message })
      }
    )
  ] });
};
const labels = {
  specialCharacter: "1 special character",
  containsNumber: "1 number",
  containsUppercase: "1 upper case",
  containsLowercase: "1 lower case",
  eightCharacter: "At least 8 characters"
};
const PasswordCriteria = ({ criteria }) => {
  return /* @__PURE__ */ jsx("div", { className: "flex flex-wrap gap-2", children: Object.entries(criteria).map(([key, isValid]) => /* @__PURE__ */ jsxs(
    "div",
    {
      className: `${isValid ? "bg-green-50 text-green-800" : "bg-slate-50 text-slate-800"} flex gap-2 items-center w-fit rounded-full px-2.5 pr-3 py-1`,
      children: [
        /* @__PURE__ */ jsx(
          "div",
          {
            className: `border-2 ${isValid ? "border-green-800" : "border-gray-800"} w-5 h-5 rounded-full flex justify-center items-center`,
            children: /* @__PURE__ */ jsx(Check, { size: 16 })
          }
        ),
        labels[key]
      ]
    },
    key
  )) });
};
const buttonVariants = cva(
  "inline-flex items-center justify-center gap-2 whitespace-nowrap rounded-md text-sm font-medium transition-colors focus-visible:outline-none focus-visible:ring-1 focus-visible:ring-ring disabled:pointer-events-none disabled:opacity-50 [&_svg]:pointer-events-none [&_svg]:size-4 [&_svg]:shrink-0",
  {
    variants: {
      variant: {
        default: "bg-primary text-primary-foreground shadow hover:bg-primary/90",
        destructive: "bg-destructive text-destructive-foreground shadow-sm hover:bg-destructive/90",
        outline: "border border-input bg-background shadow-sm hover:bg-accent hover:text-accent-foreground",
        secondary: "bg-secondary text-secondary-foreground shadow-sm hover:bg-secondary/80",
        ghost: "hover:bg-accent hover:text-accent-foreground",
        link: "text-primary underline-offset-4 hover:underline"
      },
      size: {
        default: "h-9 px-4 py-2",
        sm: "h-8 rounded-md px-3 text-xs",
        lg: "h-12 rounded-md px-8 text-lg",
        icon: "h-9 w-9",
        link: "px-0 py-0 text-base"
      }
    },
    defaultVariants: {
      variant: "default",
      size: "default"
    }
  }
);
const Button = React.forwardRef(
  ({
    className,
    variant,
    size,
    text,
    asChild = false,
    isLoading = false,
    ...props
  }, ref) => {
    const Comp = asChild ? Slot : "button";
    return /* @__PURE__ */ jsx(
      Comp,
      {
        className: cn(buttonVariants({ variant, size, className })),
        disabled: isLoading,
        ref,
        ...props,
        children: isLoading ? /* @__PURE__ */ jsxs(Fragment, { children: [
          /* @__PURE__ */ jsx(Loader2, { className: "animate-spin" }),
          "Please wait"
        ] }) : text
      }
    );
  }
);
Button.displayName = "Button";
const contactusApi = apiSlice.injectEndpoints({
  endpoints: (builder) => ({
    SignUp: builder.mutation({
      query: (SubmittedData) => ({
        url: `/auth/register/`,
        method: "POST",
        body: SubmittedData
      })
    }),
    verifyOtp: builder.mutation({
      query: (SubmittedData) => ({
        url: `/auth/registration/verify-email/`,
        method: "POST",
        body: SubmittedData
      })
    }),
    resendOtp: builder.mutation({
      query: (SubmittedData) => ({
        url: `/auth/registration/resend-email/`,
        method: "POST",
        body: SubmittedData
      })
    }),
    forgotPassword: builder.mutation({
      query: (SubmittedData) => ({
        url: `/auth/password/reset/`,
        method: "POST",
        body: SubmittedData
      })
    }),
    resetPassword: builder.mutation({
      query: (SubmittedData) => ({
        url: `/auth/password/reset/confirm/`,
        method: "POST",
        body: SubmittedData
      })
    })
  })
});
const {
  useSignUpMutation,
  useResendOtpMutation,
  useForgotPasswordMutation,
  useResetPasswordMutation
} = contactusApi;
const Dialog = ({ isOpen, onClose, children }) => /* @__PURE__ */ jsx(
  DialogPrimitive.Root,
  {
    open: isOpen,
    onOpenChange: (open) => !open && onClose(),
    children: /* @__PURE__ */ jsxs(DialogPortal, { children: [
      /* @__PURE__ */ jsx(DialogOverlay, {}),
      /* @__PURE__ */ jsxs(DialogContent, { children: [
        /* @__PURE__ */ jsx(DialogPrimitive.DialogTitle, {}),
        /* @__PURE__ */ jsx(DialogPrimitive.Description, {}),
        children
      ] })
    ] })
  }
);
const DialogPortal = DialogPrimitive.Portal;
const DialogOverlay = React.forwardRef(({ className, ...props }, ref) => /* @__PURE__ */ jsx(
  DialogPrimitive.Overlay,
  {
    ref,
    className: cn(
      "fixed inset-0 z-50 bg-black/40  data-[state=open]:animate-in data-[state=closed]:animate-out data-[state=closed]:fade-out-0 data-[state=open]:fade-in-0",
      className
    ),
    ...props
  }
));
DialogOverlay.displayName = DialogPrimitive.Overlay.displayName;
const DialogContent = React.forwardRef(({ className, children, onClose, ...props }, ref) => /* @__PURE__ */ jsxs(
  DialogPrimitive.Content,
  {
    ref,
    className: cn(
      "fixed left-[50%] top-[50%] z-50 grid w-full max-w-md translate-x-[-50%] translate-y-[-50%] gap-4 border bg-background p-6 shadow-lg duration-200 data-[state=open]:animate-in data-[state=closed]:animate-out data-[state=closed]:fade-out-0 data-[state=open]:fade-in-0 data-[state=closed]:zoom-out-95 data-[state=open]:zoom-in-95 data-[state=closed]:slide-out-to-left-1/2 data-[state=closed]:slide-out-to-top-[48%] data-[state=open]:slide-in-from-left-1/2 data-[state=open]:slide-in-from-top-[48%] sm:rounded-lg",
      className
    ),
    ...props,
    children: [
      children,
      onClose && /* @__PURE__ */ jsxs(DialogPrimitive.Close, { className: "absolute right-4 top-4 rounded-sm opacity-70 ring-offset-background transition-opacity hover:opacity-100 focus:outline-none focus:ring-2 focus:ring-ring focus:ring-offset-2 disabled:pointer-events-none data-[state=open]:bg-accent data-[state=open]:text-muted-foreground", children: [
        /* @__PURE__ */ jsx(X, { className: "h-6 w-6" }),
        /* @__PURE__ */ jsx("span", { className: "sr-only", children: "Close" })
      ] })
    ]
  }
));
DialogContent.displayName = DialogPrimitive.Content.displayName;
const checkPasswordCriteria$1 = (password) => ({
  specialCharacter: /[#?!@$%^&*-]/.test(password),
  containsUppercase: /[A-Z]/.test(password),
  containsLowercase: /[a-z]/.test(password),
  containsNumber: /[0-9]/.test(password),
  eightCharacter: password.length >= 8
});
const Signup$1 = () => {
  const [passwordCriteria, setPasswordCriteria] = useState({
    specialCharacter: false,
    containsUppercase: false,
    containsLowercase: false,
    containsNumber: false,
    eightCharacter: false
  });
  const [isOpen, setIsOpen] = useState(false);
  const navigate = useNavigate();
  const { toast: toast2 } = useToast();
  const { uid, token } = useParams();
  const methods = useForm();
  const { handleSubmit, watch } = methods;
  const password = watch("new_password1");
  const [resetPassword, { isLoading }] = useResetPasswordMutation();
  useEffect(() => {
    setPasswordCriteria(checkPasswordCriteria$1(password || ""));
  }, [password]);
  const validatePassword = (value) => {
    const criteria = checkPasswordCriteria$1(value);
    if (!criteria.eightCharacter) {
      return "Password must be at least 8 characters long.";
    }
    if (!criteria.containsUppercase) {
      return "Password must contain at least one uppercase letter.";
    }
    if (!criteria.containsLowercase) {
      return "Password must contain at least one lowercase letter.";
    }
    if (!criteria.containsNumber) {
      return "Password must contain at least one number.";
    }
    if (!criteria.specialCharacter) {
      return "Password must contain at least one special character.";
    }
    return true;
  };
  const onSubmit = (data) => {
    resetPassword({
      ...data,
      uid,
      token
    }).unwrap().then((res) => {
      console.log(res);
      if (res == null ? void 0 : res.success) {
        setIsOpen(true);
      }
    }).catch((err) => {
      var _a, _b, _c, _d, _e, _f, _g, _h, _i;
      if ((_b = (_a = err == null ? void 0 : err.data) == null ? void 0 : _a.data) == null ? void 0 : _b.non_field_errors) {
        toast2({
          title: "An error occurred",
          description: (_d = (_c = err == null ? void 0 : err.data) == null ? void 0 : _c.data) == null ? void 0 : _d.token[0],
          variant: "destructive",
          icon: /* @__PURE__ */ jsx(
            TriangleAlert,
            {
              color: "red",
              size: 35,
              className: "bg-red-100 p-1 rounded"
            }
          )
        });
      } else if ((_e = err == null ? void 0 : err.data) == null ? void 0 : _e.data) {
        toast2({
          title: "An error occurred",
          description: ((_g = (_f = err == null ? void 0 : err.data) == null ? void 0 : _f.data) == null ? void 0 : _g.uid[0]) ? (_i = (_h = err == null ? void 0 : err.data) == null ? void 0 : _h.data) == null ? void 0 : _i.uid[0] : "Something went wrong, please try again",
          variant: "destructive",
          icon: /* @__PURE__ */ jsx(
            TriangleAlert,
            {
              color: "red",
              size: 35,
              className: "bg-red-100 p-1 rounded"
            }
          )
        });
      } else {
        toast2({
          title: "An error occurred",
          description: "Something went wrong, please try again",
          variant: "destructive",
          icon: /* @__PURE__ */ jsx(
            TriangleAlert,
            {
              color: "red",
              size: 35,
              className: "bg-red-100 p-1 rounded"
            }
          )
        });
      }
    });
    console.log(data);
  };
  return /* @__PURE__ */ jsxs(AuthLayout, { title: "", subtitle: "", children: [
    /* @__PURE__ */ jsx(FormProvider, { ...methods, children: /* @__PURE__ */ jsxs(
      "form",
      {
        className: "flex flex-col gap-5 w-full",
        onSubmit: handleSubmit(onSubmit),
        children: [
          /* @__PURE__ */ jsx(
            Heading,
            {
              title: "Change Password",
              subtitle: "Enter new password and confirm password to reset your forgotten password"
            }
          ),
          /* @__PURE__ */ jsx(
            FormElement,
            {
              validateFn: validatePassword,
              type: "password",
              fieldName: "new_password1",
              fieldLabel: "New Password",
              isValidated: true,
              required: true
            }
          ),
          /* @__PURE__ */ jsx(PasswordCriteria, { criteria: passwordCriteria }),
          /* @__PURE__ */ jsx(
            FormElement,
            {
              validateFn: validatePassword,
              type: "password",
              fieldName: "new_password2",
              fieldLabel: "Confrim Password",
              isValidated: true,
              required: true
            }
          ),
          /* @__PURE__ */ jsx(Button, { size: "lg", text: "Reset Password", isLoading }),
          /* @__PURE__ */ jsx(Link, { to: "/user/login", className: "text-primary mx-auto", children: "Back To Login" })
        ]
      }
    ) }),
    /* @__PURE__ */ jsx(Dialog, { isOpen, children: /* @__PURE__ */ jsxs("div", { className: "space-y-4", children: [
      /* @__PURE__ */ jsx(
        CircleCheck,
        {
          color: "green",
          size: 35,
          className: "bg-green-100 p-2 rounded-lg absolute top-6"
        }
      ),
      /* @__PURE__ */ jsxs("div", { children: [
        /* @__PURE__ */ jsx("h2", { className: "text-2xl font-semibold", children: "Password has successfully reset" }),
        /* @__PURE__ */ jsx("p", { className: "text-sm text-muted-foreground", children: "Your password has been successfully reset. You can now login with your new password." })
      ] }),
      /* @__PURE__ */ jsx("hr", {}),
      /* @__PURE__ */ jsx(
        Button,
        {
          variant: "default",
          className: "w-full",
          onClick: () => {
            navigate("/user/login");
          },
          text: "Proceed to login"
        }
      )
    ] }) })
  ] });
};
const route1 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: Signup$1
}, Symbol.toStringTag, { value: "Module" }));
const AuthHeading$3 = {
  title: "",
  subtitle: ""
};
const MainHeading$3 = {
  title: "Account Verified Successfully",
  subtitle: "Your account has been verified successfully"
};
const OTP = () => {
  const navigate = useNavigate();
  return /* @__PURE__ */ jsxs(AuthLayout, { title: AuthHeading$3.title, subtitle: AuthHeading$3.subtitle, children: [
    /* @__PURE__ */ jsx(Heading, { title: MainHeading$3.title, subtitle: MainHeading$3.subtitle }),
    /* @__PURE__ */ jsx(
      Button,
      {
        size: "lg",
        text: "Go to login",
        className: "w-full my-4",
        onClick: () => {
          navigate("/user/login");
        }
      }
    )
  ] });
};
const route2 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: OTP
}, Symbol.toStringTag, { value: "Module" }));
const AuthHeading$2 = {
  title: "",
  subtitle: ""
};
const MainHeading$2 = {
  title: "Reset Password",
  subtitle: "A reset link will be sent to your email"
};
const ResetPassword = () => {
  const navigate = useNavigate();
  const [isOpen, setIsOpen] = useState(false);
  const [email, setEmail] = useState("");
  const methods = useForm();
  const { handleSubmit } = methods;
  const [forgotPassword, { isLoading }] = useForgotPasswordMutation();
  const onSubmit = (data) => {
    forgotPassword(data).unwrap().then((res) => {
      if (res == null ? void 0 : res.success) {
        setEmail(data == null ? void 0 : data.email);
        setIsOpen(true);
      }
    }).catch((err) => {
      var _a, _b, _c, _d, _e, _f, _g, _h, _i, _j, _k;
      if ((_b = (_a = err == null ? void 0 : err.data) == null ? void 0 : _a.data) == null ? void 0 : _b.non_field_errors) {
        toast({
          title: "An error occurred",
          description: (_d = (_c = err == null ? void 0 : err.data) == null ? void 0 : _c.data) == null ? void 0 : _d.non_field_errors[0],
          variant: "destructive",
          icon: /* @__PURE__ */ jsx(
            TriangleAlert,
            {
              color: "red",
              size: 35,
              className: "bg-red-100 p-1 rounded"
            }
          )
        });
      } else if ((_e = err == null ? void 0 : err.data) == null ? void 0 : _e.data) {
        toast({
          title: "An error occurred",
          description: ((_g = (_f = err == null ? void 0 : err.data) == null ? void 0 : _f.data) == null ? void 0 : _g.username[0]) ? (_i = (_h = err == null ? void 0 : err.data) == null ? void 0 : _h.data) == null ? void 0 : _i.username[0] : (_k = (_j = err == null ? void 0 : err.data) == null ? void 0 : _j.data) == null ? void 0 : _k.email[0],
          variant: "destructive",
          icon: /* @__PURE__ */ jsx(
            TriangleAlert,
            {
              color: "red",
              size: 35,
              className: "bg-red-100 p-1 rounded"
            }
          )
        });
      } else {
        toast({
          title: "An error occurred",
          description: "Something went wrong, please try again",
          variant: "destructive",
          icon: /* @__PURE__ */ jsx(
            TriangleAlert,
            {
              color: "red",
              size: 35,
              className: "bg-red-100 p-1 rounded"
            }
          )
        });
      }
    });
  };
  return /* @__PURE__ */ jsxs(AuthLayout, { title: AuthHeading$2.title, subtitle: AuthHeading$2.subtitle, children: [
    /* @__PURE__ */ jsxs(FormProvider, { ...methods, children: [
      /* @__PURE__ */ jsxs(
        "form",
        {
          className: "flex flex-col gap-5 w-full",
          onSubmit: handleSubmit(onSubmit),
          children: [
            /* @__PURE__ */ jsx(Heading, { title: MainHeading$2.title, subtitle: MainHeading$2.subtitle }),
            /* @__PURE__ */ jsx(FormElement, { type: "email", fieldName: "email", required: true }),
            /* @__PURE__ */ jsx(Button, { size: "lg", text: "Send Reset Link", isLoading })
          ]
        }
      ),
      /* @__PURE__ */ jsxs("p", { className: "flex gap-2 justify-center items-center font-semibold my-4", children: [
        "Not you?",
        /* @__PURE__ */ jsx(Link, { to: "/user/login", className: "text-primary", children: "Log in" })
      ] })
    ] }),
    /* @__PURE__ */ jsx(Dialog, { isOpen, children: /* @__PURE__ */ jsxs("div", { className: "space-y-4", children: [
      /* @__PURE__ */ jsx(
        CircleCheck,
        {
          color: "green",
          size: 35,
          className: "bg-green-100 p-2 rounded-lg absolute top-6"
        }
      ),
      /* @__PURE__ */ jsxs("div", { children: [
        /* @__PURE__ */ jsx("h2", { className: "text-2xl font-semibold", children: "Reset password link sent" }),
        /* @__PURE__ */ jsxs("p", { className: "text-sm text-muted-foreground", children: [
          "A reset passsword link has been sent to ",
          email
        ] })
      ] }),
      /* @__PURE__ */ jsx("hr", {}),
      /* @__PURE__ */ jsx(
        Button,
        {
          variant: "default",
          className: "w-full",
          onClick: () => {
            navigate("/user/login");
          },
          text: "Go to login"
        }
      )
    ] }) })
  ] });
};
const route3 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: ResetPassword
}, Symbol.toStringTag, { value: "Module" }));
const AuthHeading$1 = {
  title: "Manage OnePrint with ease",
  subtitle: "Log in to access the admin dashboard, track user activity, and oversee operations"
};
const MainHeading$1 = {
  title: "Login",
  subtitle: "Hey there! Welcome back 👋"
};
const Login$1 = () => {
  const methods = useForm();
  const { handleSubmit } = methods;
  const onSubmit = (data) => console.log(data);
  return /* @__PURE__ */ jsx(AuthLayout, { title: AuthHeading$1.title, subtitle: AuthHeading$1.subtitle, children: /* @__PURE__ */ jsx(FormProvider, { ...methods, children: /* @__PURE__ */ jsxs(
    "form",
    {
      className: "flex flex-col gap-5 w-full",
      onSubmit: handleSubmit(onSubmit),
      children: [
        /* @__PURE__ */ jsx(Heading, { title: MainHeading$1.title, subtitle: MainHeading$1.subtitle }),
        /* @__PURE__ */ jsx(
          FormElement,
          {
            type: "email",
            fieldName: "email",
            fieldLabel: "Work Email",
            required: true
          }
        ),
        /* @__PURE__ */ jsx(FormElement, { type: "password", fieldName: "password", required: true }),
        /* @__PURE__ */ jsx(
          Link,
          {
            to: "/forgot-password",
            className: "font-semibold self-end hover:text-primary",
            children: "Forgot your password?"
          }
        ),
        /* @__PURE__ */ jsx(Button, { size: "lg", text: "Log in" }),
        /* @__PURE__ */ jsxs("p", { className: "flex gap-2 justify-center font-semibold", children: [
          "Don't have an account?",
          " ",
          /* @__PURE__ */ jsx(Link, { to: "/admin/signup", className: "text-primary", children: "Create account" })
        ] })
      ]
    }
  ) }) });
};
const route4 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: Login$1
}, Symbol.toStringTag, { value: "Module" }));
const SocialAuth = ({ displayText }) => {
  return /* @__PURE__ */ jsxs(Fragment, { children: [
    /* @__PURE__ */ jsx("div", { className: "relative text-center text-sm after:absolute after:inset-0 after:top-1/2 after:z-0 after:flex after:items-center after:border-t after:border-border border-dotted", children: /* @__PURE__ */ jsx("span", { className: "relative z-10 bg-background px-2 text-muted-foreground", children: displayText }) }),
    /* @__PURE__ */ jsxs("div", { className: "flex justify-evenly", children: [
      /* @__PURE__ */ jsx(Link, { to: "https://accounts.google.com/signin/oauth/error/v2?authError=ChVyZWRpcmVjdF91cmlfbWlzbWF0Y2gSsAEKWW91IGNhbid0IHNpZ24gaW4gdG8gdGhpcyBhcHAgYmVjYXVzZSBpdCBkb2Vzbid0IGNvbXBseSB3aXRoIEdvb2dsZSdzIE9BdXRoIDIuMCBwb2xpY3kuCgpJZiB5b3UncmUgdGhlIGFwcCBkZXZlbG9wZXIsIHJlZ2lzdGVyIHRoZSByZWRpcmVjdCBVUkkgaW4gdGhlIEdvb2dsZSBDbG91ZCBDb25zb2xlLgogIBptaHR0cHM6Ly9kZXZlbG9wZXJzLmdvb2dsZS5jb20vaWRlbnRpdHkvcHJvdG9jb2xzL29hdXRoMi93ZWItc2VydmVyI2F1dGhvcml6YXRpb24tZXJyb3JzLXJlZGlyZWN0LXVyaS1taXNtYXRjaCCQAypMCgxyZWRpcmVjdF91cmkSPGh0dHBzOi8vb25lcHJpbnQuYXRvbXVzLmJ1enovYWNjb3VudHMvZ29vZ2xlL2xvZ2luL2NhbGxiYWNrLzKkAggBErABCllvdSBjYW4ndCBzaWduIGluIHRvIHRoaXMgYXBwIGJlY2F1c2UgaXQgZG9lc24ndCBjb21wbHkgd2l0aCBHb29nbGUncyBPQXV0aCAyLjAgcG9saWN5LgoKSWYgeW91J3JlIHRoZSBhcHAgZGV2ZWxvcGVyLCByZWdpc3RlciB0aGUgcmVkaXJlY3QgVVJJIGluIHRoZSBHb29nbGUgQ2xvdWQgQ29uc29sZS4KICAabWh0dHBzOi8vZGV2ZWxvcGVycy5nb29nbGUuY29tL2lkZW50aXR5L3Byb3RvY29scy9vYXV0aDIvd2ViLXNlcnZlciNhdXRob3JpemF0aW9uLWVycm9ycy1yZWRpcmVjdC11cmktbWlzbWF0Y2g%3D&client_id=138661767005-15rfkgmkt179jfmpvauh6nr771418dcj.apps.googleusercontent.com&flowName=GeneralOAuthFlow", children: /* @__PURE__ */ jsx("img", { src: svgs.google, alt: "Google" }) }),
      /* @__PURE__ */ jsx(Link, { to: "", children: /* @__PURE__ */ jsx("img", { src: svgs.facebook, alt: "Facebook" }) }),
      /* @__PURE__ */ jsx(Link, { to: "", children: /* @__PURE__ */ jsx("img", { src: svgs.apple, alt: "Apple" }) })
    ] })
  ] });
};
const checkPasswordCriteria = (password) => ({
  specialCharacter: /[#?!@$%^&*-]/.test(password),
  containsUppercase: /[A-Z]/.test(password),
  containsLowercase: /[a-z]/.test(password),
  containsNumber: /[0-9]/.test(password),
  eightCharacter: password.length >= 8
});
const Signup = () => {
  const { toast: toast2 } = useToast();
  const navigate = useNavigate();
  const [email, setEmail] = useState("");
  const [isOpen, setIsOpen] = useState(false);
  const [timer, setTimer] = useState(900);
  const [isDisabled, setIsDisabled] = useState(true);
  const [passwordCriteria, setPasswordCriteria] = useState({
    specialCharacter: false,
    containsUppercase: false,
    containsLowercase: false,
    containsNumber: false,
    eightCharacter: false
  });
  const methods = useForm();
  const { handleSubmit, watch } = methods;
  const password = watch("password");
  useEffect(() => {
    if (timer > 0) {
      const countdown = setInterval(() => {
        setTimer((prev) => prev - 1);
      }, 1e3);
      return () => clearInterval(countdown);
    } else {
      setIsDisabled(false);
    }
  }, [timer]);
  const [resendOtp, { isLoading: resendOtpLoading }] = useResendOtpMutation();
  const [signup, { isLoading }] = useSignUpMutation();
  useEffect(() => {
    setPasswordCriteria(checkPasswordCriteria(password || ""));
  }, [password]);
  const validatePassword = (value) => {
    const criteria = checkPasswordCriteria(value);
    if (!criteria.eightCharacter) {
      return "Password must be at least 8 characters long.";
    }
    if (!criteria.containsUppercase) {
      return "Password must contain at least one uppercase letter.";
    }
    if (!criteria.containsLowercase) {
      return "Password must contain at least one lowercase letter.";
    }
    if (!criteria.containsNumber) {
      return "Password must contain at least one number.";
    }
    if (!criteria.specialCharacter) {
      return "Password must contain at least one special character.";
    }
    return true;
  };
  const formatTime = (time) => {
    const minutes = Math.floor(time / 60);
    const seconds = time % 60;
    return `${minutes}:${seconds < 10 ? `0${seconds}` : seconds}`;
  };
  const handleResendOtp = () => {
    resendOtp({ email });
    setTimer(900);
    setIsDisabled(true);
  };
  const onSubmit = (data) => {
    signup(data).unwrap().then((res) => {
      var _a;
      if (res == null ? void 0 : res.success) {
        setEmail(data == null ? void 0 : data.email);
        setIsOpen(true);
        toast2({
          title: `Account created successfully`,
          description: (_a = res == null ? void 0 : res.data) == null ? void 0 : _a.detail,
          variant: "success",
          icon: /* @__PURE__ */ jsx(
            CircleCheck,
            {
              color: "green",
              size: 35,
              className: "bg-green-100 p-1 rounded"
            }
          )
        });
      }
    }).catch((err) => {
      var _a, _b, _c, _d, _e, _f, _g, _h, _i, _j, _k;
      if ((_b = (_a = err == null ? void 0 : err.data) == null ? void 0 : _a.data) == null ? void 0 : _b.non_field_errors) {
        toast2({
          title: "An error occurred",
          description: (_d = (_c = err == null ? void 0 : err.data) == null ? void 0 : _c.data) == null ? void 0 : _d.non_field_errors[0],
          variant: "destructive",
          icon: /* @__PURE__ */ jsx(
            TriangleAlert,
            {
              color: "red",
              size: 35,
              className: "bg-red-100 p-1 rounded"
            }
          )
        });
      } else if ((_e = err == null ? void 0 : err.data) == null ? void 0 : _e.data) {
        toast2({
          title: "An error occurred",
          description: ((_g = (_f = err == null ? void 0 : err.data) == null ? void 0 : _f.data) == null ? void 0 : _g.username[0]) ? (_i = (_h = err == null ? void 0 : err.data) == null ? void 0 : _h.data) == null ? void 0 : _i.username[0] : (_k = (_j = err == null ? void 0 : err.data) == null ? void 0 : _j.data) == null ? void 0 : _k.email[0],
          variant: "destructive",
          icon: /* @__PURE__ */ jsx(
            TriangleAlert,
            {
              color: "red",
              size: 35,
              className: "bg-red-100 p-1 rounded"
            }
          )
        });
      } else {
        toast2({
          title: "An error occurred",
          description: "Something went wrong, please try again",
          variant: "destructive",
          icon: /* @__PURE__ */ jsx(
            TriangleAlert,
            {
              color: "red",
              size: 35,
              className: "bg-red-100 p-1 rounded"
            }
          )
        });
      }
    });
  };
  return /* @__PURE__ */ jsxs(
    AuthLayout,
    {
      title: "Let's make something awesome together",
      subtitle: "Sign up to start creating, customizing, and shopping for personalized products. It's super easy!",
      children: [
        /* @__PURE__ */ jsx(FormProvider, { ...methods, children: /* @__PURE__ */ jsxs(
          "form",
          {
            className: "flex flex-col gap-5 w-full",
            onSubmit: handleSubmit(onSubmit),
            children: [
              /* @__PURE__ */ jsx(Heading, { title: "Sign Up", subtitle: "Let's go! Create an account ✨" }),
              /* @__PURE__ */ jsxs("div", { className: "flex gap-5", children: [
                /* @__PURE__ */ jsx(FormElement, { type: "text", fieldName: "first_name", required: true }),
                /* @__PURE__ */ jsx(FormElement, { type: "text", fieldName: "last_name", required: true })
              ] }),
              /* @__PURE__ */ jsxs("div", { className: "flex gap-5", children: [
                /* @__PURE__ */ jsx(FormElement, { type: "text", fieldName: "username", required: true }),
                /* @__PURE__ */ jsx(FormElement, { type: "phone", fieldName: "phone_number", required: true })
              ] }),
              /* @__PURE__ */ jsx(FormElement, { type: "email", fieldName: "email", required: true }),
              /* @__PURE__ */ jsx(
                FormElement,
                {
                  validateFn: validatePassword,
                  type: "password",
                  fieldName: "password",
                  isValidated: true,
                  required: true
                }
              ),
              /* @__PURE__ */ jsx(PasswordCriteria, { criteria: passwordCriteria }),
              /* @__PURE__ */ jsx(Button, { size: "lg", text: "Sign Up", isLoading }),
              /* @__PURE__ */ jsx(SocialAuth, { displayText: "or sign up with" }),
              /* @__PURE__ */ jsxs("p", { className: "flex gap-2 justify-center mb-10 font-semibold", children: [
                "Have an account already?",
                " ",
                /* @__PURE__ */ jsx(Link, { to: "/user/login", className: "text-primary", children: "Log in" })
              ] })
            ]
          }
        ) }),
        /* @__PURE__ */ jsx(Dialog, { isOpen, children: /* @__PURE__ */ jsxs("div", { className: "space-y-4", children: [
          /* @__PURE__ */ jsx(
            CircleCheck,
            {
              color: "green",
              size: 35,
              className: "bg-green-100 p-2 rounded-lg absolute top-6"
            }
          ),
          /* @__PURE__ */ jsxs("div", { children: [
            /* @__PURE__ */ jsx("h2", { className: "text-2xl font-semibold", children: "Verification email sent" }),
            /* @__PURE__ */ jsxs("p", { className: "text-sm text-muted-foreground", children: [
              "A verification email has been sent to ",
              email
            ] }),
            /* @__PURE__ */ jsxs("p", { className: "flex gap-2 text-sm text-muted-foreground my-4", children: [
              "Didn't the link?",
              /* @__PURE__ */ jsx(
                Button,
                {
                  onClick: handleResendOtp,
                  variant: "link",
                  size: "link",
                  className: "text-sm",
                  text: resendOtpLoading ? "Sending, please wait..." : isDisabled ? `Resend in ${formatTime(timer)}` : "Send a new code",
                  disabled: isDisabled || isLoading
                }
              )
            ] })
          ] }),
          /* @__PURE__ */ jsx("hr", {}),
          /* @__PURE__ */ jsx(
            Button,
            {
              variant: "default",
              className: "w-full",
              onClick: () => {
                navigate("/user/login");
              },
              text: "Go to login"
            }
          )
        ] }) })
      ]
    }
  );
};
const route5 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: Signup
}, Symbol.toStringTag, { value: "Module" }));
const login = apiSlice.injectEndpoints({
  endpoints: (builder) => ({
    login: builder.mutation({
      query: (credentials) => ({
        url: "/auth/login/",
        method: "POST",
        body: { ...credentials }
      }),
      transformResponse(response) {
        return response.data;
      },
      async onQueryStarted(_, { dispatch: dispatch2, queryFulfilled }) {
        try {
          const { data } = await queryFulfilled;
          dispatch2(setCredentials(data));
        } catch (err) {
          console.log(err);
        }
      }
    })
  })
});
const { useLoginMutation } = login;
const AuthHeading = {
  title: "Ready to pick up where you left off?",
  subtitle: "Log in to unleash your creativity, access your designs, orders, and keep personalizing your products"
};
const MainHeading = {
  title: "Login",
  subtitle: "Hey there! Welcome back 👋"
};
const Login = () => {
  const navigate = useNavigate();
  const methods = useForm();
  const { handleSubmit } = methods;
  const { toast: toast2 } = useToast();
  const [login2, { isLoading }] = useLoginMutation();
  const onSubmit = (data) => {
    login2(data).unwrap().then((res) => {
      console.log(res);
      toast2({
        title: "Success",
        description: "You have successfully logged in",
        variant: "success",
        icon: /* @__PURE__ */ jsx(
          CircleCheck,
          {
            color: "green",
            size: 35,
            className: "bg-green-100 p-1 rounded"
          }
        )
      });
      if (window.history.state && window.history.state.idx > 0) {
        navigate(-1);
      } else {
        navigate("/");
      }
    }).catch((err) => {
      var _a, _b, _c, _d, _e;
      if ((_a = err == null ? void 0 : err.data) == null ? void 0 : _a.data) {
        toast2({
          title: "An error occurred",
          description: ((_c = (_b = err == null ? void 0 : err.data) == null ? void 0 : _b.data) == null ? void 0 : _c.non_field_errors[0]) ? (_e = (_d = err == null ? void 0 : err.data) == null ? void 0 : _d.data) == null ? void 0 : _e.non_field_errors[0] : "Something went wrong, please try again",
          variant: "destructive",
          icon: /* @__PURE__ */ jsx(
            TriangleAlert,
            {
              color: "red",
              size: 35,
              className: "bg-red-100 p-1 rounded"
            }
          )
        });
      } else {
        toast2({
          title: "An error occurred",
          description: "Something went wrong, please try again",
          variant: "destructive",
          icon: /* @__PURE__ */ jsx(
            TriangleAlert,
            {
              color: "red",
              size: 35,
              className: "bg-red-100 p-1 rounded"
            }
          )
        });
      }
    });
  };
  return /* @__PURE__ */ jsx(AuthLayout, { title: AuthHeading.title, subtitle: AuthHeading.subtitle, children: /* @__PURE__ */ jsx(FormProvider, { ...methods, children: /* @__PURE__ */ jsxs(
    "form",
    {
      className: "flex flex-col gap-5 w-full",
      onSubmit: handleSubmit(onSubmit),
      children: [
        /* @__PURE__ */ jsx(Heading, { title: MainHeading.title, subtitle: MainHeading.subtitle }),
        /* @__PURE__ */ jsx(FormElement, { type: "email", fieldName: "email", required: true }),
        /* @__PURE__ */ jsx(FormElement, { type: "password", fieldName: "password", required: true }),
        /* @__PURE__ */ jsx(
          Link,
          {
            to: "/forgot-password",
            className: "font-semibold self-end hover:text-primary",
            children: "Forgot your password?"
          }
        ),
        /* @__PURE__ */ jsx(Button, { size: "lg", text: "Log in", isLoading }),
        /* @__PURE__ */ jsx(SocialAuth, { displayText: "or login with" }),
        /* @__PURE__ */ jsxs("p", { className: "flex gap-2 justify-center font-semibold mb-10", children: [
          "Don't have an account?",
          " ",
          /* @__PURE__ */ jsx(Link, { to: "/user/signup", className: "text-primary", children: "Create account" })
        ] })
      ]
    }
  ) }) });
};
const route6 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: Login
}, Symbol.toStringTag, { value: "Module" }));
const Dashboard = () => {
  return /* @__PURE__ */ jsx("div", { children: "Dashboard" });
};
const route7 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: Dashboard
}, Symbol.toStringTag, { value: "Module" }));
const DefaultLayout = ({ children }) => {
  const [discount, setDiscount] = useState(true);
  const sections = [
    {
      title: "Company",
      links: ["About", "Features", "Works", "Career"]
    },
    {
      title: "Help",
      links: [
        "Customer Support",
        "Delivery Details",
        "Terms & Conditions",
        "Privacy Policy"
      ]
    },
    {
      title: "FAQ",
      links: ["Account", "Manage Deliveries", "Orders", "Payments"]
    },
    {
      title: "Resources",
      links: [
        "Free eBooks",
        "Development Tutorial",
        "How to - Blog",
        "Youtube Playlist"
      ]
    }
  ];
  return /* @__PURE__ */ jsxs("main", { className: "relative", children: [
    /* @__PURE__ */ jsxs("div", { className: "fixed z-50 w-full bg-white shadow-lg", children: [
      discount && /* @__PURE__ */ jsxs("div", { className: "flex text-center gap-2 bg-black text-white py-2", children: [
        /* @__PURE__ */ jsxs("p", { className: "w-full", children: [
          "Sign up and get 20% off to your first order.",
          " ",
          /* @__PURE__ */ jsx(Link, { className: "font-semibold border-b-2", to: "/user/signup", children: "Sign Up Now" })
        ] }),
        /* @__PURE__ */ jsx(
          "button",
          {
            onClick: () => setDiscount(false),
            className: "w-full max-w-32",
            children: /* @__PURE__ */ jsx(X, {})
          }
        )
      ] }),
      /* @__PURE__ */ jsxs("div", { className: "max-w-8xl mx-auto px-10 py-5 flex items-center", children: [
        /* @__PURE__ */ jsx("div", { className: "w-full", children: /* @__PURE__ */ jsx("img", { src: svgs.logo, alt: "Profile Images", className: "h-10 w-fit" }) }),
        /* @__PURE__ */ jsxs("div", { className: "w-full flex items-center gap-14", children: [
          /* @__PURE__ */ jsxs("div", { className: "flex gap-2 items-center w-full bg-slate-100 rounded-md", children: [
            /* @__PURE__ */ jsx("span", { className: "px-4 py-3", children: /* @__PURE__ */ jsx(Search, { color: "#989898" }) }),
            /* @__PURE__ */ jsx(
              "input",
              {
                placeholder: "I’m looking for...",
                className: "bg-transparent"
              }
            )
          ] }),
          /* @__PURE__ */ jsxs("div", { className: "flex gap-5", children: [
            /* @__PURE__ */ jsx("button", { className: "hover:bg-slate-100 p-2 rounded-md", children: /* @__PURE__ */ jsx(Heart, { size: 30 }) }),
            /* @__PURE__ */ jsx("button", { className: "hover:bg-slate-100 p-2 rounded-md", children: /* @__PURE__ */ jsx(User, { size: 30 }) }),
            /* @__PURE__ */ jsxs("button", { className: "relative hover:bg-slate-100 p-2 rounded-md", children: [
              /* @__PURE__ */ jsx(ShoppingBasket, { size: 30 }),
              /* @__PURE__ */ jsx("div", { className: "absolute right-1 top-7 flex items-center justify-center text-white text-[10px] bg-primary rounded-full h-4 w-4", children: "0" })
            ] })
          ] })
        ] })
      ] })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "absolute left-0 top-32 w-full h-screen", children: [
      children,
      /* @__PURE__ */ jsxs("footer", { className: "bg-gray-100 py-10", children: [
        /* @__PURE__ */ jsxs("div", { className: "container mx-auto px-6 lg:px-20 grid grid-cols-1 md:grid-cols-5 gap-6", children: [
          /* @__PURE__ */ jsxs("div", { className: "md:col-span-1", children: [
            /* @__PURE__ */ jsx("h3", { className: "text-3xl font-bold mb-4", children: "OnePrint" }),
            /* @__PURE__ */ jsx("p", { className: "text-gray-600 leading-relaxed mb-10 max-w-40", children: "High-quality, customizable products to bring your brand or event to life." }),
            /* @__PURE__ */ jsxs("div", { className: "flex space-x-4", children: [
              /* @__PURE__ */ jsx("a", { href: "#", className: "text-gray-600 hover:text-gray-800", children: /* @__PURE__ */ jsx(Twitter, {}) }),
              /* @__PURE__ */ jsx("a", { href: "#", className: "text-gray-600 hover:text-gray-800", children: /* @__PURE__ */ jsx(Facebook, {}) }),
              /* @__PURE__ */ jsx("a", { href: "#", className: "text-gray-600 hover:text-gray-800", children: /* @__PURE__ */ jsx(Instagram, {}) }),
              /* @__PURE__ */ jsx("a", { href: "#", className: "text-gray-600 hover:text-gray-800", children: /* @__PURE__ */ jsx(Github, {}) })
            ] })
          ] }),
          sections.map((section, index) => /* @__PURE__ */ jsxs("div", { className: "space-y-6 md:col-span-1", children: [
            /* @__PURE__ */ jsx("h4", { className: "text-lg font-semibold mb-4 uppercase", children: section.title }),
            /* @__PURE__ */ jsx("ul", { className: "space-y-4 text-gray-600", children: section.links.map((link, linkIndex) => /* @__PURE__ */ jsx("li", { children: /* @__PURE__ */ jsx(Link, { to: "#", className: "hover:text-gray-800", children: link }) }, linkIndex)) })
          ] }, index))
        ] }),
        /* @__PURE__ */ jsx("hr", { className: "mb-6 mt-14 border-gray-300" }),
        /* @__PURE__ */ jsxs("div", { className: "container mx-auto px-6 lg:px-20 flex flex-col md:flex-row justify-between items-center space-y-4 md:space-y-0", children: [
          /* @__PURE__ */ jsx("p", { className: "text-gray-600 pb-2", children: "Imprint © 2000-2024, All Rights Reserved" }),
          /* @__PURE__ */ jsxs("div", { className: "flex", children: [
            /* @__PURE__ */ jsx("img", { src: svgs.visa, alt: "Visa", className: "h-14" }),
            /* @__PURE__ */ jsx("img", { src: svgs.mastercard, alt: "Mastercard", className: "h-14" }),
            /* @__PURE__ */ jsx("img", { src: svgs.paypal, alt: "PayPal", className: "h-14" }),
            /* @__PURE__ */ jsx("img", { src: svgs.applepay, alt: "Apple Pay", className: "h-14" }),
            /* @__PURE__ */ jsx("img", { src: svgs.googlepay, alt: "Google Pay", className: "h-14" })
          ] })
        ] })
      ] })
    ] })
  ] });
};
const testimonial = [
  {
    name: "Sarah M.",
    text: `"I'm blown away by the quality and style of the clothes I received from Imprint. From casual wear to elegant dresses, every piece I've bought has exceeded my expectations.”`
  },
  {
    name: "James L.",
    text: `"As someone who's always on the lookout for unique fashion pieces, I'm thrilled to have stumbled upon Imprint. The selection of clothes is not only diverse but also on-point with the latest trends.”`
  },
  {
    name: "Alex K.",
    text: `"Finding clothes that align with my personal style used to be a challenge until I discovered Imprint. The range of options they offer is truly remarkable, catering to a variety of tastes and occasions.”`
  },
  {
    name: "Alex K.",
    text: `"Finding clothes that align with my personal style used to be a challenge until I discovered Imprint. The range of options they offer is truly remarkable, catering to a variety of tastes and occasions.”`
  },
  {
    name: "James L.",
    text: `"As someone who's always on the lookout for unique fashion pieces, I'm thrilled to have stumbled upon Imprint. The selection of clothes is not only diverse but also on-point with the latest trends.”`
  }
];
const logos = [
  svgs.makeWish,
  svgs.nascar,
  svgs.amazon,
  svgs.oracle,
  svgs.comedyCentral
];
const products = [
  {
    image: images.product1,
    name: "Printed Wristbands",
    price: 25.52
  },
  {
    image: images.product2,
    name: "sunglasses",
    price: 30.08
  },
  {
    image: images.product3,
    name: "classic stick pens",
    price: 10
  },
  {
    image: images.product4,
    name: "pvc luggage tags",
    price: 20.01
  },
  {
    image: images.product5,
    name: "sport bottle",
    price: 22.2
  },
  {
    image: images.product6,
    name: "cup sleeves",
    price: 10.02
  },
  {
    image: images.product7,
    name: "reliever balls",
    price: 10.05
  },
  {
    image: images.product8,
    name: "stationery",
    price: 18
  },
  {
    image: images.product9,
    name: "woven lanyards",
    price: 20.25
  }
];
const productCategory = [
  {
    image: images.category1,
    name: "Apparel"
  },
  {
    image: images.category2,
    name: "Bags"
  },
  {
    image: images.category3,
    name: "Writting"
  },
  {
    image: images.category4,
    name: "Drinkware"
  },
  {
    image: images.category5,
    name: "Outdoor"
  },
  {
    image: images.category6,
    name: "Keychains"
  },
  {
    image: images.category7,
    name: "Stationery"
  },
  {
    image: images.category8,
    name: "ID & Lanyards"
  }
];
const bgImages = [
  {
    url: images.bg1,
    position: "center",
    size: "cover",
    color: "#1414134D"
  },
  {
    url: images.bg2,
    position: "bottom right",
    size: "cover",
    color: "#1414134D"
  },
  {
    url: images.bg3,
    position: "center",
    size: "cover",
    color: "#1414134D"
  },
  {
    url: images.bg4,
    position: "center",
    size: "cover",
    color: "#1414134D"
  },
  {
    url: images.bg5,
    position: "center",
    size: "contain",
    color: "#141413"
  }
];
const Category = ({ productCategory: productCategory2 }) => {
  return /* @__PURE__ */ jsx("div", { className: "grid grid-cols-4 gap-x-6 gap-y-8", children: productCategory2.map((category, index) => /* @__PURE__ */ jsxs(
    "div",
    {
      className: "flex flex-col gap-2 text-center group hover:cursor-pointer",
      children: [
        /* @__PURE__ */ jsx(
          "img",
          {
            src: category.image,
            alt: category.name,
            className: "mx-auto transform transition-transform duration-300 group-hover:scale-105"
          }
        ),
        /* @__PURE__ */ jsxs("p", { className: "flex items-center gap-2 w-fit mx-auto text-lg", children: [
          category.name,
          " ",
          /* @__PURE__ */ jsx("span", { className: "hidden group-hover:block transition-opacity", children: /* @__PURE__ */ jsx(ArrowRight, {}) })
        ] })
      ]
    },
    index
  )) });
};
const BestProducts = ({ image, name, price }) => {
  const [hearted, setHearted] = useState(false);
  return /* @__PURE__ */ jsxs("div", { className: "relative flex flex-col gap-2 text-center group hover:cursor-pointer", children: [
    /* @__PURE__ */ jsx("div", { className: "absolute right-5 top-5 p-2 rounded-full bg-white w-fit", children: /* @__PURE__ */ jsx(
      Heart,
      {
        onClick: () => setHearted(!hearted),
        size: 20,
        color: hearted ? "none" : "black",
        fill: hearted ? "red" : "none"
      }
    ) }),
    /* @__PURE__ */ jsx("img", { src: image, alt: name, className: "mx-auto h-96 w-full rounded-2xl" }),
    /* @__PURE__ */ jsx(
      Button,
      {
        text: "Customize Now",
        className: "invisible translate-y-5 group-hover:translate-y-0 group-hover:visible transform transition-transform duration-300 absolute bottom-20 left-7 px-28 py-6 text-lg mx-auto"
      }
    ),
    /* @__PURE__ */ jsxs("div", { children: [
      /* @__PURE__ */ jsx("p", { className: "text-left text-lg uppercase text-slate-400", children: name }),
      /* @__PURE__ */ jsxs("p", { className: "text-left font-semibold", children: [
        "$",
        price,
        " (per piece)"
      ] })
    ] })
  ] });
};
const Section = ({ children, heading }) => {
  return /* @__PURE__ */ jsxs("section", { className: "flex flex-col gap-8 max-w-7xl mx-auto", children: [
    /* @__PURE__ */ jsx("h3", { className: "text-center font-semibold text-3xl", children: heading }),
    children
  ] });
};
const SimpleCard = ({
  heading,
  content,
  color
}) => /* @__PURE__ */ jsxs("div", { className: `flex flex-col gap-3 px-6 py-10 rounded-lg ${color}`, children: [
  /* @__PURE__ */ jsx("h4", { className: "text-xl font-semibold", children: heading }),
  /* @__PURE__ */ jsx("h5", { children: content })
] });
const Testimonial = ({
  testimonial: testimonial2
}) => {
  return /* @__PURE__ */ jsx("div", { className: "flex gap-5 my-3 py-2 overflow-x-auto", children: testimonial2.map((testimony, index) => /* @__PURE__ */ jsxs(
    "div",
    {
      className: "flex flex-col gap-3 p-5 border-2 rounded-xl",
      children: [
        /* @__PURE__ */ jsxs("div", { className: "flex gap-2", children: [
          /* @__PURE__ */ jsx(Star, { size: 28, color: "none", fill: "#FFC633" }),
          /* @__PURE__ */ jsx(Star, { size: 28, color: "none", fill: "#FFC633" }),
          /* @__PURE__ */ jsx(Star, { size: 28, color: "none", fill: "#FFC633" }),
          /* @__PURE__ */ jsx(Star, { size: 28, color: "none", fill: "#FFC633" }),
          /* @__PURE__ */ jsx(Star, { size: 28, color: "none", fill: "#FFC633" })
        ] }),
        /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-1", children: [
          /* @__PURE__ */ jsx("p", { className: "text-xl font-bold", children: testimony.name }),
          /* @__PURE__ */ jsx(
            "svg",
            {
              width: "20",
              height: "21",
              viewBox: "0 0 20 21",
              fill: "none",
              xmlns: "http://www.w3.org/2000/svg",
              children: /* @__PURE__ */ jsx(
                "path",
                {
                  d: "M10 0.829102C8.07164 0.829102 6.18657 1.40093 4.58319 2.47227C2.97982 3.54362 1.73013 5.06636 0.992179 6.84794C0.254225 8.62952 0.061142 10.5899 0.437348 12.4812C0.813554 14.3725 1.74215 16.1098 3.10571 17.4734C4.46928 18.837 6.20656 19.7656 8.09787 20.1418C9.98919 20.518 11.9496 20.3249 13.7312 19.5869C15.5127 18.849 17.0355 17.5993 18.1068 15.9959C19.1782 14.3925 19.75 12.5075 19.75 10.5791C19.7473 7.99408 18.7192 5.51571 16.8913 3.68783C15.0634 1.85994 12.585 0.831831 10 0.829102ZM14.2806 8.85973L9.03063 14.1097C8.96097 14.1795 8.87826 14.2348 8.78721 14.2725C8.69616 14.3103 8.59857 14.3297 8.5 14.3297C8.40144 14.3297 8.30385 14.3103 8.2128 14.2725C8.12175 14.2348 8.03903 14.1795 7.96938 14.1097L5.71938 11.8597C5.57865 11.719 5.49959 11.5281 5.49959 11.3291C5.49959 11.1301 5.57865 10.9392 5.71938 10.7985C5.86011 10.6577 6.05098 10.5787 6.25 10.5787C6.44903 10.5787 6.6399 10.6577 6.78063 10.7985L8.5 12.5188L13.2194 7.79848C13.2891 7.72879 13.3718 7.67352 13.4628 7.63581C13.5539 7.59809 13.6515 7.57868 13.75 7.57868C13.8486 7.57868 13.9461 7.59809 14.0372 7.63581C14.1282 7.67352 14.2109 7.72879 14.2806 7.79848C14.3503 7.86816 14.4056 7.95088 14.4433 8.04193C14.481 8.13297 14.5004 8.23056 14.5004 8.3291C14.5004 8.42765 14.481 8.52523 14.4433 8.61627C14.4056 8.70732 14.3503 8.79004 14.2806 8.85973Z",
                  fill: "#01AB31"
                }
              )
            }
          )
        ] }),
        /* @__PURE__ */ jsx("p", { className: "text-lg text-slate-500 min-w-96", children: testimony.text })
      ]
    },
    index
  )) });
};
const meta = () => {
  return [
    { title: "New Remix App" },
    { name: "description", content: "Welcome to Remix!" }
  ];
};
function Index() {
  const [currentImage, setCurrentImage] = useState(0);
  useEffect(() => {
    const interval = setInterval(() => {
      setCurrentImage((prevImage) => (prevImage + 1) % bgImages.length);
    }, 5e3);
    return () => clearInterval(interval);
  }, [bgImages == null ? void 0 : bgImages.length]);
  return /* @__PURE__ */ jsxs(DefaultLayout, { children: [
    /* @__PURE__ */ jsxs("section", { className: "flex flex-col", children: [
      /* @__PURE__ */ jsx(
        "div",
        {
          className: "relative bg-auto w-full h-screen py-12 px-3 flex items-end",
          style: {
            backgroundImage: `url(${bgImages[currentImage].url})`,
            backgroundPosition: bgImages[currentImage].position,
            backgroundSize: bgImages[currentImage].size,
            backgroundRepeat: "no-repeat",
            backgroundColor: bgImages[currentImage].color,
            filter: "grayscale(30%)"
          },
          children: /* @__PURE__ */ jsxs("div", { className: "bg-white bg-opacity-10 backdrop-blur-md backdrop-contrast-100 flex flex-col gap-3 max-w-[37rem] p-6 rounded-md shadow-sm absolute left-12 bottom-36", children: [
            /* @__PURE__ */ jsx("h2", { className: "bg-white bg-opacity-30 backdrop-blur-sm rounded-full font-semibold w-fit text-gray-700 px-3 py-1", children: "Create Products That Inspire." }),
            /* @__PURE__ */ jsx("h1", { className: "font-semibold text-6xl leading-tight text-white", children: "Make It Yours, Make it Memorable" }),
            /* @__PURE__ */ jsx("p", { className: "my-3 text-lg font-semibold text-gray-700", children: "With endless options and fast delivery, creating custom products has never been easier." }),
            /* @__PURE__ */ jsx(
              Button,
              {
                className: "py-6 text-md px-8 w-fit",
                text: "Start Customizing"
              }
            )
          ] })
        }
      ),
      /* @__PURE__ */ jsxs("div", { className: "flex justify-evenly bg-[#F5F5F5] py-3.5", children: [
        /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3 font-semibold text-lg", children: [
          /* @__PURE__ */ jsx(Clock, { size: 40 }),
          "1 week turn around"
        ] }),
        /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3 font-semibold text-lg", children: [
          /* @__PURE__ */ jsx(FileText, { size: 40 }),
          "Printed in America"
        ] }),
        /* @__PURE__ */ jsxs("div", { className: "flex items-center gap-3 font-semibold text-lg", children: [
          /* @__PURE__ */ jsx(Truck, { size: 40 }),
          "Free Delivery ",
          /* @__PURE__ */ jsx("span", { className: "text-sm -mx-1.5", children: "over $75" })
        ] })
      ] })
    ] }),
    /* @__PURE__ */ jsxs("div", { className: "flex flex-col gap-24 py-14", children: [
      /* @__PURE__ */ jsxs("section", { className: "flex flex-col gap-12 overflow-hidden", children: [
        /* @__PURE__ */ jsx("div", { className: "relative text-center text-sm after:absolute after:inset-0 after:top-1/2 after:z-0 after:flex after:items-center after:border-t after:border-border", children: /* @__PURE__ */ jsx("span", { className: "relative z-10 bg-background px-2 text-black text-xl font-semibold", children: "Trusted by over 1,000,000 Customers" }) }),
        /* @__PURE__ */ jsxs("div", { className: "flex gap-24 animate-scroll whitespace-nowrap", children: [
          logos.map((logo2, index) => /* @__PURE__ */ jsx(
            "img",
            {
              src: logo2,
              alt: `Trusted brand ${index + 1}`,
              className: "h-16 w-auto"
            },
            index
          )),
          logos.map((logo2, index) => /* @__PURE__ */ jsx(
            "img",
            {
              src: logo2,
              alt: `Trusted brand duplicate ${index + 1}`,
              className: "h-16 w-auto"
            },
            `duplicate-${index}`
          ))
        ] }),
        /* @__PURE__ */ jsx("hr", {})
      ] }),
      /* @__PURE__ */ jsx(Section, { heading: "Explore Categories", children: /* @__PURE__ */ jsx(Category, { productCategory }) }),
      /* @__PURE__ */ jsxs(Section, { heading: "Best Selling Products", children: [
        /* @__PURE__ */ jsx("div", { className: "grid grid-cols-3 gap-x-6 gap-y-8", children: products == null ? void 0 : products.map((product, index) => /* @__PURE__ */ jsx(
          BestProducts,
          {
            image: product.image,
            name: product.name,
            price: product.price
          },
          index
        )) }),
        /* @__PURE__ */ jsx(
          Button,
          {
            text: "View More",
            className: "w-full max-w-60 py-6 text-lg mx-auto mt-8"
          }
        )
      ] }),
      /* @__PURE__ */ jsx(Section, { heading: "Shop Featured Brands", children: /* @__PURE__ */ jsx("div", { className: "flex justify-between flex-wrap gap-4", children: Array.from({ length: 12 }, (_, index) => /* @__PURE__ */ jsx("div", { className: "min-w-40", children: /* @__PURE__ */ jsx(
        "img",
        {
          src: images[`brand${index + 1}`],
          alt: `Brand ${index + 1}`
        }
      ) }, index)) }) }),
      /* @__PURE__ */ jsxs(Section, { heading: "Why Clients Choose Us", children: [
        /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-2 gap-5", children: [
          /* @__PURE__ */ jsx(
            SimpleCard,
            {
              heading: "Quality",
              color: "bg-[#F0F5F9]",
              content: "Our products are made to stand out and last. We focus on exceptional materials and meticulous printing methods to ensure that each item meets the highest standards of durability and aesthetics."
            }
          ),
          /* @__PURE__ */ jsx(
            SimpleCard,
            {
              heading: "Flexibility",
              color: "bg-[#FDBA48]",
              content: "We provide you with complete creative control—from product selection to color and design. Our intuitive tools make it easy to bring your unique vision to life, however complex or simple it may be."
            }
          )
        ] }),
        /* @__PURE__ */ jsxs("div", { className: "grid grid-cols-3 gap-5", children: [
          /* @__PURE__ */ jsx(
            SimpleCard,
            {
              heading: "Reliability",
              color: "bg-[#EAE8ED]",
              content: "Your time and satisfaction are our priorities. From design to doorstep, we make sure each order is handled with accuracy, care, and efficiency, so you can count on dependable service every time."
            }
          ),
          /* @__PURE__ */ jsx(
            SimpleCard,
            {
              heading: "Support",
              color: "bg-[#BCEDED]",
              content: "Our customer service team is here for you, whether you need help with design choices, placing an order, or tracking a shipment. We’re committed to assisting you at every step until you’re completely satisfied."
            }
          ),
          /* @__PURE__ */ jsx(
            SimpleCard,
            {
              heading: "Value",
              color: "bg-[#EAE8ED]",
              content: "Get premium quality without the premium price. We’re committed to offering fair pricing, whether you need a single item or a bulk order, so you never have to compromise on quality or budget."
            }
          )
        ] })
      ] }),
      /* @__PURE__ */ jsx(Section, { heading: "What Our Customers Say", children: /* @__PURE__ */ jsx(Testimonial, { testimonial }) })
    ] })
  ] });
}
const route8 = /* @__PURE__ */ Object.freeze(/* @__PURE__ */ Object.defineProperty({
  __proto__: null,
  default: Index,
  meta
}, Symbol.toStringTag, { value: "Module" }));
const serverManifest = { "entry": { "module": "/assets/entry.client-Dkr78KKH.js", "imports": ["/assets/jsx-runtime-BMrMXMSG.js", "/assets/components-Cr-DqYPJ.js", "/assets/index-XrU1i-Uc.js"], "css": [] }, "routes": { "root": { "id": "root", "parentId": void 0, "path": "", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/root-B12AXFqz.js", "imports": ["/assets/jsx-runtime-BMrMXMSG.js", "/assets/components-Cr-DqYPJ.js", "/assets/index-XrU1i-Uc.js", "/assets/authSlice-wB7ecGB_.js", "/assets/use-toast-8lerf42F.js", "/assets/index-CukTLwcq.js", "/assets/utils-Br2EXFOw.js", "/assets/x-BXrg1M6t.js"], "css": ["/assets/root-CQfMIWRD.css"] }, "routes/auth.reset.$uid.$token": { "id": "routes/auth.reset.$uid.$token", "parentId": "root", "path": "auth/reset/:uid/:token", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/auth.reset._uid._token-CT3WVaWJ.js", "imports": ["/assets/jsx-runtime-BMrMXMSG.js", "/assets/index-BvM5axzC.js", "/assets/Heading-XaJD9TaV.js", "/assets/PasswordCriteria-KH_F8Z_3.js", "/assets/button-B6K71EBC.js", "/assets/dialog-CvN3Q5Ey.js", "/assets/use-toast-8lerf42F.js", "/assets/index-XrU1i-Uc.js", "/assets/components-Cr-DqYPJ.js", "/assets/triangle-alert-DbGJ6_AS.js", "/assets/utils-Br2EXFOw.js", "/assets/authSlice-wB7ecGB_.js", "/assets/index-CukTLwcq.js", "/assets/x-BXrg1M6t.js"], "css": ["/assets/index-DpsRipQV.css"] }, "routes/account-verification": { "id": "routes/account-verification", "parentId": "root", "path": "account-verification", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/account-verification-J5TGj5HY.js", "imports": ["/assets/jsx-runtime-BMrMXMSG.js", "/assets/button-B6K71EBC.js", "/assets/Heading-XaJD9TaV.js", "/assets/index-XrU1i-Uc.js", "/assets/utils-Br2EXFOw.js", "/assets/authSlice-wB7ecGB_.js"], "css": [] }, "routes/forgot-password": { "id": "routes/forgot-password", "parentId": "root", "path": "forgot-password", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/forgot-password-D8qjwCYp.js", "imports": ["/assets/jsx-runtime-BMrMXMSG.js", "/assets/button-B6K71EBC.js", "/assets/Heading-XaJD9TaV.js", "/assets/index-BvM5axzC.js", "/assets/dialog-CvN3Q5Ey.js", "/assets/use-toast-8lerf42F.js", "/assets/index-XrU1i-Uc.js", "/assets/components-Cr-DqYPJ.js", "/assets/triangle-alert-DbGJ6_AS.js", "/assets/utils-Br2EXFOw.js", "/assets/authSlice-wB7ecGB_.js", "/assets/index-CukTLwcq.js", "/assets/x-BXrg1M6t.js"], "css": ["/assets/index-DpsRipQV.css"] }, "routes/admin.login": { "id": "routes/admin.login", "parentId": "root", "path": "admin/login", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/admin.login-Gjmar8bA.js", "imports": ["/assets/jsx-runtime-BMrMXMSG.js", "/assets/button-B6K71EBC.js", "/assets/Heading-XaJD9TaV.js", "/assets/index-BvM5axzC.js", "/assets/components-Cr-DqYPJ.js", "/assets/utils-Br2EXFOw.js", "/assets/authSlice-wB7ecGB_.js", "/assets/index-XrU1i-Uc.js"], "css": ["/assets/index-DpsRipQV.css"] }, "routes/user.signup": { "id": "routes/user.signup", "parentId": "root", "path": "user/signup", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/user.signup-D_8zeHMn.js", "imports": ["/assets/jsx-runtime-BMrMXMSG.js", "/assets/index-BvM5axzC.js", "/assets/dialog-CvN3Q5Ey.js", "/assets/Heading-XaJD9TaV.js", "/assets/SocialAuth-khL-ycbD.js", "/assets/PasswordCriteria-KH_F8Z_3.js", "/assets/button-B6K71EBC.js", "/assets/use-toast-8lerf42F.js", "/assets/index-XrU1i-Uc.js", "/assets/components-Cr-DqYPJ.js", "/assets/triangle-alert-DbGJ6_AS.js", "/assets/utils-Br2EXFOw.js", "/assets/index-CukTLwcq.js", "/assets/x-BXrg1M6t.js", "/assets/authSlice-wB7ecGB_.js"], "css": ["/assets/index-DpsRipQV.css"] }, "routes/user.login": { "id": "routes/user.login", "parentId": "root", "path": "user/login", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/user.login-DBkYQeOg.js", "imports": ["/assets/jsx-runtime-BMrMXMSG.js", "/assets/button-B6K71EBC.js", "/assets/Heading-XaJD9TaV.js", "/assets/index-BvM5axzC.js", "/assets/SocialAuth-khL-ycbD.js", "/assets/use-toast-8lerf42F.js", "/assets/authSlice-wB7ecGB_.js", "/assets/index-XrU1i-Uc.js", "/assets/components-Cr-DqYPJ.js", "/assets/triangle-alert-DbGJ6_AS.js", "/assets/utils-Br2EXFOw.js"], "css": ["/assets/index-DpsRipQV.css"] }, "routes/dashboard": { "id": "routes/dashboard", "parentId": "root", "path": "dashboard", "index": void 0, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/dashboard-BVycAelA.js", "imports": ["/assets/jsx-runtime-BMrMXMSG.js"], "css": [] }, "routes/_index": { "id": "routes/_index", "parentId": "root", "path": void 0, "index": true, "caseSensitive": void 0, "hasAction": false, "hasLoader": false, "hasClientAction": false, "hasClientLoader": false, "hasErrorBoundary": false, "module": "/assets/route-DfpRDDRG.js", "imports": ["/assets/jsx-runtime-BMrMXMSG.js", "/assets/button-B6K71EBC.js", "/assets/components-Cr-DqYPJ.js", "/assets/x-BXrg1M6t.js", "/assets/utils-Br2EXFOw.js", "/assets/index-XrU1i-Uc.js"], "css": [] } }, "url": "/assets/manifest-1d6cb133.js", "version": "1d6cb133" };
const mode = "production";
const assetsBuildDirectory = "build\\client";
const basename = "/";
const future = { "v3_fetcherPersist": true, "v3_relativeSplatPath": true, "v3_throwAbortReason": true, "v3_routeConfig": false, "v3_singleFetch": true, "v3_lazyRouteDiscovery": true, "unstable_optimizeDeps": false };
const isSpaMode = false;
const publicPath = "/";
const entry = { module: entryServer };
const routes = {
  "root": {
    id: "root",
    parentId: void 0,
    path: "",
    index: void 0,
    caseSensitive: void 0,
    module: route0
  },
  "routes/auth.reset.$uid.$token": {
    id: "routes/auth.reset.$uid.$token",
    parentId: "root",
    path: "auth/reset/:uid/:token",
    index: void 0,
    caseSensitive: void 0,
    module: route1
  },
  "routes/account-verification": {
    id: "routes/account-verification",
    parentId: "root",
    path: "account-verification",
    index: void 0,
    caseSensitive: void 0,
    module: route2
  },
  "routes/forgot-password": {
    id: "routes/forgot-password",
    parentId: "root",
    path: "forgot-password",
    index: void 0,
    caseSensitive: void 0,
    module: route3
  },
  "routes/admin.login": {
    id: "routes/admin.login",
    parentId: "root",
    path: "admin/login",
    index: void 0,
    caseSensitive: void 0,
    module: route4
  },
  "routes/user.signup": {
    id: "routes/user.signup",
    parentId: "root",
    path: "user/signup",
    index: void 0,
    caseSensitive: void 0,
    module: route5
  },
  "routes/user.login": {
    id: "routes/user.login",
    parentId: "root",
    path: "user/login",
    index: void 0,
    caseSensitive: void 0,
    module: route6
  },
  "routes/dashboard": {
    id: "routes/dashboard",
    parentId: "root",
    path: "dashboard",
    index: void 0,
    caseSensitive: void 0,
    module: route7
  },
  "routes/_index": {
    id: "routes/_index",
    parentId: "root",
    path: void 0,
    index: true,
    caseSensitive: void 0,
    module: route8
  }
};
export {
  serverManifest as assets,
  assetsBuildDirectory,
  basename,
  entry,
  future,
  isSpaMode,
  mode,
  publicPath,
  routes
};
